<?php
/**
 * @class geoipxeModel
 * @author KnDol (kndol@kndol.net)
 * @brief geoipxe module's Model class
 **/

class geoipxeModel extends geoipxe {
	function init() {
	}

	/**
	 * @brief Return geoipxe module's configuration
	 */
	function getGeoipxeConfig() {
		static $Config;

		if($Config) return $Config;

		// Get member configuration stored in the DB
		$oModuleModel = getModel('module');
		$config = $oModuleModel->getModuleConfig('geoipxe');

		if(!$config->db_type) $config->db_type = 'db1';

		$Config = $config;

		return $config;
	}

	/**
	 * Make the given number positive by wrapping it to 32 bit values
	 *
	 * @access private
	 * @static
	 * @param int $x  Number to wrap
	 * @return int
	 */
	private static function wrap32($x) {
		return $x + ($x < 0 ? 4294967296 : 0);
	}
	
	/**
	 * Get the IP version and number of the given IP address
	 *
	 * This method will return an array, whose components will be:
	 * - first: 4 if the given IP address is an IPv4 one, 6 if it's an IPv6 one,
	 *          or fase if it's neither.
	 * - second: the IP address' number if its version is 4, the number string if
	 *           its version is 6, false otherwise.
	 *
	 * @access private
	 * @static
	 * @param string $ip  IP address to extract the version and number for
	 * @return array
	 */
	private static function ipVersionAndNumber($ip) {
		if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
			return array(4, sprintf('%u', ip2long($ip)));
		}
		elseif (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
			$result = 0;

			foreach (str_split(bin2hex(inet_pton($ip)), 8) as $word) {
				$result = bcadd(bcmul($result, '4294967296', 0), self::wrap32(hexdec($word)), 0);
			}

			return  array(6, $result);
		}
		else {
			// Invalid IP address, return falses
			return  array(false, false);
		}
	}

	/**
	 * @brief Return geoipxe module's configuration
	 */
	function getGeoipxeDataFromIP($ip) {
		$location = (object)array('country_code'=>'','country'=>'');
		// extract IP version and number
		list($ipVersion, $ipNumber) = self::ipVersionAndNumber($ip);
		if (!$ipNumber) return $location;
		$config = self::getGeoipxeConfig();

		$oDB = &DB::getInstance();
		$table = "geoipxe_".$config->db_type."_v".$ipVersion;
		$xe_table = $oDB->prefix.$table;
		$columns = $config->db_type == "db1" ? "a.country_code, a.country" : "a.country_code, a.country, a.region, a.city";
		switch($oDB->db_type)
		{
			case 'mysql':
			case 'mysql_innodb':
			case 'mysqli':
			case 'mysqli_innodb':
				$query = "SELECT ".$columns." FROM (SELECT `ip_from` FROM `".$xe_table."` AS `".$table."` WHERE `ip_from` <= ".$ipNumber." and `ip_to` >= ".$ipNumber." ORDER BY `ip_from` DESC LIMIT 1) `b` JOIN `".$xe_table."` `a` on b.ip_from = a.ip_from;";
				break;
			case 'mssql':
				$query = "SELECT ".$columns." FROM (SELECT ip_from FROM ".$xe_table." AS ".$table." WHERE ip_from <= ".$ipNumber." and ip_to >= ".$ipNumber." ORDER BY ip_from DESC) b JOIN ".$xe_table." a on b.ip_from = a.ip_from;";
				break;
			case 'cubrid':
				$query = 'SELECT '.$columns.' FROM (SELECT "ip_from" FROM "'.$xe_table.'" AS "'.$table.'" WHERE "ip_from" <= '.$ipNumber.' and "ip_to" >= '.$ipNumber.' ORDER BY "ip_from" DESC LIMIT 1) "b" JOIN "'.$xe_table.'" "a" on b.ip_from = a.ip_from;';
				break;
		}
		$output = $oDB->_query($query);
		$location = $oDB->_fetch($output);

		return $location;
	}
}
/* End of file geoipxe.controller.php */
/* Location: ./modules/geoipxe/geoipxe.model.php */
