<?php
/**
 * @class  qstatisticsModel
 * @author NHN (developers@xpressengine.com)
 * @brief Model class of the qstatistics module
 */

class qstatisticsModel extends qstatistics {

	var $analyticsreporting = NULL;
	var $viewID = NULL;
	
	/**
	 * Initialization
	 * @return void
	 */
	function init()
	{
		$oModuleModel = getModel('module');
		
		$config = $oModuleModel->getModuleConfig( $this->module );
		if( empty($config->google_analytics_viewID) || !is_file( $config->google_service_account_key_filename ) || !is_file( $config->google_api_php_client_path . '/autoload.php' ) ) return;
		
		if( !empty($this->analyticsreporting) ) return;
		
		// Load the Google API PHP Client Library.
		require_once( $config->google_api_php_client_path . '/autoload.php' );
		
		$this->analyticsreporting = $this->initializeAnalyticsReporting( $config );
		$this->viewID = $config->google_analytics_viewID;
	}
	
	/**
	 * Initializes an Analytics Reporting API V4 service object.
	 *
	 * @return An authorized Analytics Reporting API V4 service object.
	 */
	function initializeAnalyticsReporting( $config )
	{
		if( empty($config->google_analytics_viewID) || !is_file( $config->google_service_account_key_filename ) || !is_file( $config->google_api_php_client_path . '/autoload.php' ) ) return NULL;
		
		// Use the developers console and download your service account
		// credentials in JSON format. Place them in this directory or
		// change the key file location if necessary.
		$KEY_FILE_LOCATION = $config->google_service_account_key_filename;
		
		// Create and configure a new client object.
		$client = new Google_Client();
		$client->setApplicationName("Google Analytics Reporting");
		$client->setAuthConfig($KEY_FILE_LOCATION);
		$client->setScopes(['https://www.googleapis.com/auth/analytics.readonly']);
		$analyticsreporting = new Google_Service_AnalyticsReporting($client);
		
		return $analyticsreporting;
	}

	/**
	 * Parses and prints the Analytics Reporting API V4 response.
	 *
	 * @param An Analytics Reporting API V4 response.
	 */
	function outputResults($reports) {
		$output = new stdClass();
		$output->dimesions = array();
		$output->metrics = array();

		for ( $reportIndex = 0; $reportIndex < count( $reports ); $reportIndex++ ) {
			$report = $reports[ $reportIndex ];
			$header = $report->getColumnHeader();
			$dimensionHeaders = $header->getDimensions();
			$metricHeaders = $header->getMetricHeader()->getMetricHeaderEntries();
			$rows = $report->getData()->getRows();

			for ( $rowIndex = 0; $rowIndex < count($rows); $rowIndex++) {
				$row = $rows[ $rowIndex ];
				$dimensions = $row->getDimensions();
				$metrics = $row->getMetrics();
				for ($i = 0; $i < count($dimensionHeaders) && $i < count($dimensions); $i++) {
					$output->dimesions[$dimensionHeaders[$i]] = $dimensions[$i];
				}

				for ($j = 0; $j < count($metrics); $j++) {
					$values = $metrics[$j]->getValues();
					for ($k = 0; $k < count($values); $k++) {
						$entry = $metricHeaders[$k];
						$output->metrics[$entry->getName()] = $values[$k];
					}
				}
			}
		}

		return $output;
	}

	//

	function getRows($response, $dimension_names) {
		/* get dimension headers and metric headers from the API response */
		for ($i = 0; $i < count($response); $i++) {
			$header = $response[$i]->getColumnHeader();
			$dimensionHeaders = $header->getDimensions();
			$metricHeaders = $header->getMetricHeader()->getMetricHeaderEntries();
			$rows = $response[$i]->getData()->getRows();
	 
			/* iterate row by row */
			$report_rows = array();
			for ( $rowIndex = 0; $rowIndex < count($rows); $rowIndex++) {
				$row = $rows[$rowIndex];
				$dimensions = $row->getDimensions();
				$metrics = $row->getMetrics();
				$result_row = array();
	 
				/* get dimension data */
				for ($i = 0; $i < count($dimensionHeaders) && $i < count($dimensions); $i++) {
				   $result_row[$dimension_names[$i]] = $dimensions[$i];
				}
	 
				/* get metric data */
				for ($i = 0; $i < count($metricHeaders); $i++) {
					$result_row[$metricHeaders[$i]->getName()] = $metrics[0]->getValues()[$i];
				}
	 
				/* add the row to the report */
				$report_rows[] = $result_row;
			}
		}
		return $report_rows;
	}

	//

	function getGAReports($analyticsreporting, $params) {
		if ( empty($analyticsreporting) ) return new Object(-1, 'Error: Google Analytics Reporting must be set');
		
		/* set view_id - required */
		if ( empty($this->viewID) ) return new Object(-1, 'Error: Google Analytics View ID must be set');
		
		/* set date -required */
		$dateRange = new Google_Service_AnalyticsReporting_DateRange();
		if (isset($params->startDate) && isset($params->endDate)) {
			$dateRange->setStartDate($params->startDate);
			$dateRange->setEndDate($params->endDate);
		}
		else {
			return new Object(-1, 'Error: Google Analytics date range must be set');
		}
	 
		/* Set dimensions - required */
		if (!isset($params->dimensions) || empty ($params->dimensions)) {
			return new Object(-1, 'Error: No dimensions have been set');
		}
		else {
			$dimensions = array();
			$dimension_names = array();
			foreach ($params->dimensions as $k => $v) {
				$dimension = new Google_Service_AnalyticsReporting_Dimension();
				$dimension->setName($k);
				$dimensions[] = $dimension;
				$dimension_names[] = $v;
			}
		}
	 
		/* set metrics - required */
		if (!isset($params->metrics) || empty ($params->metrics)) {
			return new Object(-1, 'Error: No metrics have been set');
		}
		else {
			$metrics = array();
			$metric_names = array();
			foreach ($params->metrics as $k => $v) {
				$metric = new Google_Service_AnalyticsReporting_Metric();
				$metric->setExpression($k);
				$metric->setAlias($v);
				$metrics[] = $metric;
				$metric_names[] = $v;
			}
		}
	 
		/* set dimension filter - optional */
		if (isset($params->dimensionFilter) && isset($params->dimensionFilter['dimensionName']) && isset($params->dimensionFilter['operator']) && isset($params->dimensionFilter['expression'])) {
			$dimensionFilter = new Google_Service_AnalyticsReporting_DimensionFilter();
			$dimensionFilter->setDimensionName($params->dimensionFilter['dimensionName']);
			$dimensionFilter->setOperator($params->dimensionFilter['operator']);
			$dimensionFilter->setExpressions($params->dimensionFilter['expression']);
			if (isset($params->dimensionFilter['exclude'] ))
				$dimensionFilter->setNot($params->dimensionFilter['exclude']);
			$dimensionFilterClause = new Google_Service_AnalyticsReporting_DimensionFilterClause();
			$dimensionFilterClause->setFilters($dimensionFilter);
		}
		
		// create the metric filter
		if (isset($params->metricFilter) && isset($params->metricFilter['metricName']) && isset($params->metricFilter['operator']) && isset($params->metricFilter['value'])) {
			$metricFilterSessions = new Google_Service_AnalyticsReporting_MetricFilter();
			$metricFilterSessions->setMetricName($params->metricFilter['metricName']);
			$metricFilterSessions->setOperator($params->metricFilter['operator']);
			$metricFilterSessions->setComparisonValue($params->metricFilter['value']);
			if (isset($params->metricFilter['exclude'] ))
				$metricFilter->setNot($params->metricFilter['exclude']);
			$metricFilterClause = new Google_Service_AnalyticsReporting_MetricFilterClause();
			$metricFilterClause->setFilters([$metricFilterSessions]);
		}
		
		/* set order by - optional */
		if (isset($params->orderField)) {
			$orderby = new Google_Service_AnalyticsReporting_OrderBy();
			$orderby->setFieldName($params->orderField);
			if (isset( $params->orderType))
				$orderby->setSortOrder($params->orderType);
		 }
	 
	 
	 
	 
		/* Create the ReportRequest object */
		$request = new Google_Service_AnalyticsReporting_ReportRequest();
		$request->setViewId($this->viewID);
		$request->setDateRanges($dateRange);
		$request->setDimensions($dimensions);
		$request->setMetrics($metrics);
		if (isset($dimensionFilterClause))
			 $request->setDimensionFilterClauses($dimensionFilterClause);
		if (isset($metricFilterClause))
			 $request->setMetricFilterClauses($metricFilterClause);
		if (isset($orderby))
			$request->setOrderBys($orderby);
		if (isset($params->pageSize))
			$request->setPageSize($params->pageSize);
	 
	 
	 
		/* set the request body */
		$body = new Google_Service_AnalyticsReporting_GetReportsRequest();
	 
	 
	 
	 
		/* iterate to call the API and store the response data, 500 a time */
		$pageToken = NULL;
		$report = array();
		
		do {
			// fire the request
			$body->setReportRequests(array($request));
			$response = $analyticsreporting->reports->batchGet($body);

			// get rows from the response
			$rows = $this->getRows($response, $dimension_names);

			// merge rows into the report
			$report = array_merge($report, $rows);

			if( !empty($params->unLimited) )
			{
				// set next page token to get the next 500 rows
				$pageToken = $response[0]->getNextPageToken();
				$request->setPageToken($pageToken);
			}

		} while ( $pageToken != null );
		
		$output = new Object();
		$output->data = $report;
		return $output;
	}
	
	//
	
	function getReports()
	{
		$vars = Context::getRequestVars();
		
		$output = $this->getGAReports( $this->analyticsreporting, $vars );
		if(!$output->toBool()) return $output;
		
		$this->add( 'reports', $output->data );
	}
	
	//
	
}
/* End of file qstatistics.model.php */
/* Location: ./modules/qstatistics/qstatistics.model.php */
