/**
 * Clevis_Poll
 *
 * 설문조사 클릭을 라디오버튼이나 체크박스가 아닌 이미지를 통해 제어하는 자바스크립트로
 * 제로보드 설문조사 모듈에 최적화 되어 있습니다.
 * 선택 갯수를 지정 복수 선택 가능
 *
 * < 기능 >
 *  - 이미지 버튼을 이용하여 라디오버튼 또는 체크박스를 제어,
 *  - 선택 가능한 갯수 표시
 *  - 선택 가능 갯수 초과시 선택한 순서대로 첫번째 선택 항목부터 선택해제
 *  - 문서를 모두 불러온뒤 선택 가능하도록 함 (오작동을 방지)
 *  - 설문 항목이 긴 경우 잘리는 뒷부분을 보여주기 위한 툴팁 지원
 *    (※ 툴팁의 경우 결과 목록은 초기화 함수에서 자동으로 이벤트를 추가하지 않음)
 *
 * < 알려진 문제 >
 *  - FireFox 에서는 공백이 없는 영문은 워드랩이 지원되지 않기 때문에 공백없는 문자열이
 *    있을 경우 툴팁이 표시 안될 수 있습니다.
 *
 * @author      KAi ( http://kais.tistory.com/ )
 * @license     MIT License
 * @version     1.2
 */
var Clevis_Poll = {
	formObjects : null,
	_Tconfig : {
		item_height : 21,// 폼에서 아이템 항목의 높이
		tooltip_minheight : 20,// 툴팁으로 보여줄 문자열의 최소 높이
		tooltip_width_fix : 32 // 툴팁을 항목의 폭과 맞추기 위해 스타일에 적용된 수치를 뺌
	},

	/**
	 * initialize : 초기화 함수
	 * @return	(void)
	 */
	Initialize : function() {

		// 폼이 여러개일 수 있기 때문에 해당 폼을 모두 가져옴
		var form = Clevis_Poll.formObjects = jQuery("form[id^='pollforms_']");

		if (typeof form !== 'object' || form.length <= 0) {
			return false;
		}

		// 설문조사 폼이 여러개일 경우, 각 폼마다 이벤트 추가
		jQuery.each(
			form,
			function(formIndex, formElement) {
				jQuery.each(
					jQuery("div .itemtext", formElement.parentNode),// 폼의 상위 객체를 이용해 선택 항목 가져옴
					function(idx, el) { jQuery(el).click(function() { Clevis_Poll.Toggle(formIndex, this); }); }
				);
			}
		);
	},

	/**
	 * GetConfig : 설정값 가져오기
	 * @param	(string)	name		설정명
	 * @return	(mixed)					설정값
	 */
	GetConfig : function(name) {
		return Clevis_Poll._Tconfig[name] || false;
	},

	/**
	 * SetConfig : 설정값 저장
	 * @param	(string)	name		설정명
	 * @param	(mixed)		value		설정값
	 * @return	(void)
	 */
	SetConfig : function(name, value) {
		switch (typeof name) {
			case 'string': Clevis_Poll._Tconfig[name] = value; break;
			case 'object': for (var n in name) { Clevis_Poll._Tconfig[n] = name[n]; }
		}
	},

	/**
	 * SetItems : 라디오버튼/체크박스 토글 함수
	 * @param	(string)	groupid		설문조사 그룹 아이디
	 * @param	(object)	values		선택할 항목들의 값
	 * @return	(void)
	 */
	SetItems : function(groupid, values, formIndex) {
		jQuery("input[name^='item_'][name$='_"+ groupid +"']", Clevis_Poll.formObjects[formIndex])
			.val(values);
	},

	/**
	 * SetButtons : 선택 항목 버튼 토글 함수
	 * @param	(string)	groupid		설문조사 그룹 아이디
	 * @param	(object)	values		선택할 항목들의 값
	 * @param	(string)	itemid		(optional) 현재 선택한 항목 값,
	 *									값이 있을 경우 클래스 값에 따라 ON/OFF 를 토글
	 * @return	(void)
	 */
	SetButtons : function(groupid, values, itemid) {
		jQuery.each(
			jQuery("div .itemtext[group='"+ groupid +"']", "div .button-list"),
			function(idx, el) {
				var currid = jQuery(el).attr('item');

				// 목록 중 객체가 현재 선택한 항목일 경우 상위 객체의 ClassName 변경
				if (itemid !== null && currid == itemid) {
					jQuery(el).parent().toggleClass('off').toggleClass('on')

				// 목록 중 객체가 선택된 항목 목록에 없을 경우 선택 해제
				// (다른 항목을 선택하면서 해제된 항목 같은 경우)
				} else if (jQuery.inArray(currid, values) === -1) {
					el.parentNode.className = el.parentNode.className.replace(' on', ' off');
				}
			}
		);
	},

	/**
	 * DisplayCount : 선택 가능한 항목 갯수 표시
	 * @param	(string)	groupid		그룹아이디
	 * @param	(string)	value		표시 값
	 * @return	(void)
	 */
	DisplayCount : function(groupid, value) {
		jQuery('#display_'+ groupid).html(value);
	},

	/**
	 * ShowTooltip : 툴팁 보여주기
	 * @param	(object)	target		대상
	 * @param	(string)	type		툴팁 타입
	 * @return	(void)
	 */
	ShowTooltip : function(target, type) {
		var parent = jQuery(target).parent().parent(),
			tooltip = document.createElement('DIV');

		// 툴팁 생성후 객체에 추가
		// [2009-05-26] fixed: pollbox 객체가 여러개 일 경우 툴팁이 나타나지 않기 때문에, 현재 타겟을 기준으로 pollbox 객체에 추가
		// [2009-05-28] fixed: 툴팁의 offset 위치 때문에 가려지거나 안보이는 문제로 인해 항목의 상위 객체에 추가
		jQuery(parent).append(tooltip);

		// 툴팁 초기화 
		jQuery(tooltip)
			.addClass('tooltip')
			.addClass('wordbreak')// fixed: 영문이 연속으로 이뤄진 글자의 경우 워드랩되지 않는 문제
			.text(jQuery(target).text())

			// [2009-05-26] fixed: 항목의 길이에 퍼센트의 길이도 포함되어 있기에 퍼센트를 제외하고 계산하기 위해 타겟의 폭을 지정
			.css('width', (jQuery(target).width()) +'px');

		// 항목의 내용이 긴 경우만 보여주기 위해 툴팁의 높이가 길어지는지 확인
		// 최소 높이보다 짧다면 객체 삭제후 리턴
		if (jQuery(tooltip).height() <= Clevis_Poll.GetConfig('tooltip_minheight')) {
			jQuery(tooltip).remove();
			return false;
		}

		// 툴팁의 폭과 위치 설정 및 Fade-in 효과 추가
		jQuery(tooltip)
			.css('width', (jQuery(target).parent().width() - Clevis_Poll.GetConfig('tooltip_width_fix')) +'px')
			// 선택항목을 가리지 않기 위해 항목 높이 더해줌
			.css('top', parseInt(jQuery(target).offset()['top'] - jQuery(parent).offset()['top'] + Clevis_Poll.GetConfig('item_height')) +'px')
			.css('left', '1px')
			.fadeIn(300);// Fade-in effect

		// 매번 생성되는 툴팁 객체를 제거하기 위해 툴팁이 표시되어 보여진 후에 mouseout() 이벤트 추가
		jQuery(target).mouseout(function() { Clevis_Poll.HideTooltip(tooltip); })
	},

	/**
	 * HideTooltip : 툴팁 숨김
	 * @param	(object)	tooltip		툴팁 객체
	 * @return	(void)
	 */
	HideTooltip : function(tooltip) {
		var Target = tooltip;
		jQuery(tooltip)
			.fadeOut(
				300,
				function() {
					// fixed: mouseout() 이벤트가 추가된 객체가 툴팁 객체가 아니므로 툴팁 객체를 삭제시 this 로 불가능
					// 별도의 변수(Target)에 툴팁 객체를 담고 해당 변수를 통해 삭제
					jQuery(Target).remove();

					// fixed: mouseout() 이벤트 삭제 추가
					jQuery(this).unbind('mouseout');
				}
			);// Fade-out effect
	},

	/**
	 * Toggle : 선택 버튼 토글 함수
	 *  라디오버튼/체크박스를 기준으로 해당 항목을 선택/선택해제
	 * @param	(element)	target		선택 버튼의 객체
	 * @return	(void)
	 */
	Toggle : function(formIndex, target) {
		var groupid = jQuery(target).attr('group'),// 설문조사 그룹 값
			itemid = jQuery(target).attr('item'),// 설문조사 항목 값

			// 선택 가능 갯수
			limit = parseInt(jQuery("input[name^='checkcount_"+ groupid +"']", Clevis_Poll.formObjects[formIndex]).val()),

			// 선택한 항목 큐, 여러개 선택 가능할 경우 선택 제한 갯수 초과시 오래된 것부터 비움
			queue = Clevis_Poll.GetConfig('queue_'+ groupid) || [];

		// 설문항목의 값이 선택되어 있는지 확인후 처리
		// 이미 선택되어 있을 경우 선택 갯수와 선택 가능한 갯수 비교,
		// 선택 가능한 갯수 초과시 선택된 순서대로 오래된 것부터 선택해제
		// 만약, 선택되어 있을 경우 큐에서 해당 항목을 찾아서 선택 해제
		if (jQuery('#item_'+ itemid).attr('checked') === false) {
			if (queue.length >= limit) {
				queue.shift();
			}
			queue.push(itemid);

		} else if (queue.length > 0) {
			queue.splice(jQuery.inArray(itemid, queue), 1);
		}

		// 큐 저장
		Clevis_Poll.SetConfig('queue_'+ groupid, queue);

		// 현재 선택 가능한 갯수를 표시
		Clevis_Poll.DisplayCount(groupid, (limit - queue.length));

		// 라디오/체크박스를 선택함
		Clevis_Poll.SetItems(groupid, queue, formIndex);

		// 큐를 기준으로 큐에 저장된 값을 제외한 항목들을 선택해제
		Clevis_Poll.SetButtons(groupid, queue, itemid);

		return false;
	}
};

jQuery(document).ready(Clevis_Poll.Initialize);