<?php
	/**
	 * File: Browser.php
	 * Author: Chris Schuld (http://chrisschuld.com/)
	 * Last Modified: November 08, 2009
	 * @version 1.6
	 * @package PegasusPHP
	 * 
	 * Copyright (C) 2008-2009 Chris Schuld  (chris@chrisschuld.com)
	 *
	 * This program is free software; you can redistribute it and/or
	 * modify it under the terms of the GNU General Public License as
	 * published by the Free Software Foundation; either version 2 of
	 * the License, or (at your option) any later version.
	 *
	 * This program is distributed in the hope that it will be useful,
	 * but WITHOUT ANY WARRANTY; without even the implied warranty of
	 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	 * GNU General Public License for more details at:
	 * http://www.gnu.org/copyleft/gpl.html
	 *
	 *
	 * Typical Usage:
	 * 
	 *   $browser = new Browser();
	 *   if( $browser->getBrowser() == Browser::BROWSER_FIREFOX && $browser->getVersion() >= 2 ) {
	 *   	echo 'You have FireFox version 2 or greater';
	 *   }
	 *
	 * User Agents Sampled from: http://www.useragentstring.com/
	 * 
	 * This implementation is based on the original work from Gary White
	 * http://apptools.com/phptools/browser/
	 * 
	 * Gary White noted: "Since browser detection is so unreliable, I am
	 * no longer maintaining this script. You are free to use and or
	 * modify/update it as you want, however the author assumes no
	 * responsibility for the accuracy of the detected values."
	 * 
	 * Anyone experienced with Gary's script might be interested in these notes:
	 * 
	 *   Added class constants
	 *   Added detection and version detection for Google's Chrome
	 *   Updated the version detection for Amaya
	 *   Updated the version detection for Firefox
	 *   Updated the version detection for Lynx
	 *   Updated the version detection for WebTV
	 *   Updated the version detection for NetPositive
	 *   Updated the version detection for IE
	 *   Updated the version detection for OmniWeb
	 *   Updated the version detection for iCab
	 *   Updated the version detection for Safari
	 *   Updated Safari to remove mobile devices (iPhone)
	 *   Added detection for iPhone
	 *   Added detection for robots
	 *   Added detection for mobile devices
	 *   Added detection for BlackBerry
	 *   Removed Netscape checks (matches heavily with firefox & mozilla)
	 * 
	 * 
	 * ADDITIONAL UPDATES:
	 * 
	 * 2008-11-07:
	 *  + Added Google's Chrome to the detection list
	 *  + Added isBrowser(string) to the list of functions special thanks to
	 *    Daniel 'mavrick' Lang for the function concept (http://mavrick.id.au)
	 * 
	 * 2008-12-09:
	 *  + Removed unused constant
	 *
	 * 2009-02-16: (Rick Hale)
	 *  + Added version detection for Android phones.
	 * 
	 * 2009-03-14:
	 *  + Added detection for iPods.
	 *  + Added Platform detection for iPhones
	 *  + Added Platform detection for iPods
	 * 
	 * 2009-04-22:
	 *  + Added detection for GoogleBot
	 *  + Added detection for the W3C Validator.
	 *  + Added detection for Yahoo! Slurp
	 * 
	 * 2009-04-27:
	 *  + Updated the IE check to remove a typo and bug (thanks John)
	 *  
	 * 2009-08-18:
	 *  + Updated to support PHP 5.3 - removed all deprecated function calls
	 *  + Updated to remove all double quotes (") -- converted to single quotes (')
	 *  
     * 2009-11-08:
     *  + PHP 5.3 Support
     *  + Added support for BlackBerry OS and BlackBerry browser
     *  + Added support for the Opera Mini browser
     *  + Added additional documenation
     *  + Added support for isRobot() and isMobile()
     *  + Added support for Opera version 10
     *  + Added support for deprecated Netscape Navigator version 9
     *  + Added support for IceCat
     *  + Added support for Shiretoko
     */

	class Browser {
		var $_agent = '';
		var $_browser_name = '';
		var $_version = '';
		var $_platform = '';
		var $_os = '';
		var $_is_aol = false;
        var $_is_mobile = false;
        var $_is_robot = false;
		var $_aol_version = '';

		var $BROWSER_UNKNOWN = 'unknown';
		var $VERSION_UNKNOWN = 'unknown';
		
		var $BROWSER_OPERA = 'Opera';                            // http://www.opera.com/
        var $BROWSER_OPERA_MINI = 'Opera Mini';                  // http://www.opera.com/mini/
		var $BROWSER_WEBTV = 'WebTV';                            // http://www.webtv.net/pc/
		var $BROWSER_IE = 'Internet Explorer';                   // http://www.microsoft.com/ie/
		var $BROWSER_POCKET_IE = 'Pocket Internet Explorer';     // http://en.wikipedia.org/wiki/Internet_Explorer_Mobile
		var $BROWSER_KONQUEROR = 'Konqueror';                    // http://www.konqueror.org/
		var $BROWSER_ICAB = 'iCab';                              // http://www.icab.de/
		var $BROWSER_OMNIWEB = 'OmniWeb';                        // http://www.omnigroup.com/applications/omniweb/
		var $BROWSER_FIREBIRD = 'Firebird';                      // http://www.ibphoenix.com/
		var $BROWSER_FIREFOX = 'Firefox';                        // http://www.mozilla.com/en-US/firefox/firefox.html
		var $BROWSER_SHIRETOKO = 'Shiretoko';                    // http://wiki.mozilla.org/Projects/shiretoko
		var $BROWSER_MOZILLA = 'Mozilla';                        // http://www.mozilla.com/en-US/
		var $BROWSER_AMAYA = 'Amaya';                            // http://www.w3.org/Amaya/
		var $BROWSER_LYNX = 'Lynx';                              // http://en.wikipedia.org/wiki/Lynx
		var $BROWSER_SAFARI = 'Safari';                          // http://apple.com
		var $BROWSER_IPHONE = 'iPhone';                          // http://apple.com
        var $BROWSER_IPOD = 'iPod';                              // http://apple.com 
		var $BROWSER_IPAD = 'iPad';                              // http://apple.com 
		var $BROWSER_CHROME = 'Chrome';                          // http://www.google.com/chrome
        var $BROWSER_ANDROID = 'Android';                        // http://www.android.com/
        var $BROWSER_GOOGLEBOT = 'GoogleBot';                    // http://en.wikipedia.org/wiki/Googlebot
        var $BROWSER_SLURP = 'Yahoo! Slurp';                     // http://en.wikipedia.org/wiki/Yahoo!_Slurp
        var $BROWSER_W3CVALIDATOR = 'W3C Validator';             // http://validator.w3.org/
        var $BROWSER_BLACKBERRY = 'BlackBerry';                  // http://www.blackberry.com/
        var $BROWSER_ICECAT = 'IceCat';                          // http://en.wikipedia.org/wiki/GNU_IceCat
		
        var $BROWSER_NETSCAPE_NAVIGATOR = 'Netscape Navigator';  // http://browser.netscape.com/ (DEPRECATED)
		var $BROWSER_GALEON = 'Galeon';                          // http://galeon.sourceforge.net/ (DEPRECATED)
		var $BROWSER_NETPOSITIVE = 'NetPositive';                // http://en.wikipedia.org/wiki/NetPositive (DEPRECATED)
		var $BROWSER_PHOENIX = 'Phoenix';                        // http://en.wikipedia.org/wiki/History_of_Mozilla_Firefox (DEPRECATED)
        
		var $PLATFORM_UNKNOWN = 'unknown';
		var $PLATFORM_WINDOWS = 'Windows';
		var $PLATFORM_WINDOWS_CE = 'Windows CE';
		var $PLATFORM_APPLE = 'Apple';
		var $PLATFORM_LINUX = 'Linux';
		var $PLATFORM_OS2 = 'OS/2';
		var $PLATFORM_BEOS = 'BeOS';
		var $PLATFORM_IPHONE = 'iPhone';
		var $PLATFORM_IPOD = 'iPod';
		var $PLATFORM_IPAD = 'iPad';
        var $PLATFORM_BLACKBERRY = 'BlackBerry';
		
		var $OPERATING_SYSTEM_UNKNOWN = 'unknown';
		
		function Browser() {
			$this->reset();
			$this->determine();
		}
		/**
		 * Reset all properties
		 */
		function reset() {
			$this->_agent = (isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : '');
			$this->_browser_name = $this->BROWSER_UNKNOWN;
			$this->_version = $this->VERSION_UNKNOWN;
			$this->_platform = $this->PLATFORM_UNKNOWN;
			$this->_os = $this->OPERATING_SYSTEM_UNKNOWN;
			$this->_is_aol = false;
            $this->_is_mobile = false;
            $this->_is_robot = false;
			$this->_aol_version = $this->VERSION_UNKNOWN;
		}
		
		/**
		 * Check to see if the specific browser is valid
		 * @param string $browserName
		 * @return True if the browser is the specified browser
		 */
		function isBrowser($browserName) { return( 0 == strcasecmp($this->_browser_name, trim($browserName))); }

		/**
		 * The name of the browser.  All return types are from the class contants
		 * @return string Name of the browser
		 */
		function getBrowser() { return $this->_browser_name; }
		/**
		 * Set the name of the browser
		 * @param $browser The name of the Browser
		 */
		function setBrowser($browser) { return $this->_browser_name = $browser; }
		/**
		 * The name of the platform.  All return types are from the class contants
		 * @return string Name of the browser
		 */
		function getPlatform() { return $this->_platform; }
		/**
		 * Set the name of the platform
		 * @param $platform The name of the Platform
		 */
		function setPlatform($platform) { return $this->_platform = $platform; }
		/**
		 * The version of the browser.
		 * @return string Version of the browser (will only contain alpha-numeric characters and a period)
		 */
		function getVersion() { return $this->_version; }
		/**
		 * Set the version of the browser
		 * @param $version The version of the Browser
		 */
		function setVersion($version) { $this->_version = preg_replace('[^0-9,.,a-z,A-Z]','',$version); }
		/**
		 * The version of AOL.
		 * @return string Version of AOL (will only contain alpha-numeric characters and a period)
		 */
		function getAolVersion() { return $this->_aol_version; }
		/**
		 * Set the version of AOL
		 * @param $version The version of AOL
		 */
		function setAolVersion($version) { $this->_aol_version = preg_replace('[^0-9,.,a-z,A-Z]','',$version); }
		/**
		 * Is the browser from AOL?
		 * @return boolean True if the browser is from AOL otherwise false
		 */
		function isAol() { return $this->_is_aol; }
		/**
		 * Is the browser from a mobile device?
		 * @return boolean True if the browser is from a mobile device otherwise false
		 */
		function isMobile() { return $this->_is_mobile; }
		/**
		 * Is the browser from a robot (ex Slurp,GoogleBot)?
		 * @return boolean True if the browser is from a robot otherwise false
		 */
		function isRobot() { return $this->_is_robot; }
		/**
		 * Set the browser to be from AOL
		 * @param $isAol
		 */
		function setAol($isAol) { $this->_is_aol = $isAol; }
		/**
		 * Get the user agent value in use to determine the browser
		 * @return string The user agent from the HTTP header
		 */
		function getUserAgent() { return $this->_agent; }
		/**
		 * Set the user agent value (the construction will use the HTTP header value - this will overwrite it)
		 * @param $agent_string The value for the User Agent
		 */
		function setUserAgent($agent_string) {
			$this->reset();
			$this->_agent = $agent_string;
			$this->determine();
		}
        function setMobile($value=true) {
            $this->_is_mobile = $value;
        }
        function setRobot($value=true) {
            $this->_is_robot = $value;
        }
		/**
		 * Protected routine to calculate and determine what the browser is in use (including platform)
		 */
		function determine() {
			$this->checkPlatform();
			$this->checkBrowsers();
			$this->checkForAol();
		}

		/**
		 * Protected routine to determine the browser type
		 * @return boolean True if the browser was detected otherwise false
		 */
		function checkBrowsers() {
			return (
						$this->checkBrowserGoogleBot() ||
						$this->checkBrowserSlurp() ||
						$this->checkBrowserInternetExplorer() ||
						$this->checkBrowserShiretoko() ||
						$this->checkBrowserIceCat() ||
						$this->checkBrowserNetscapeNavigator9Plus() ||
						$this->checkBrowserFirefox() ||
						$this->checkBrowserChrome() ||
                        $this->checkBrowserAndroid() ||
						$this->checkBrowserSafari() ||
						$this->checkBrowserOpera() ||
						$this->checkBrowserNetPositive() ||
						$this->checkBrowserFirebird() ||
						$this->checkBrowserGaleon() ||
						$this->checkBrowserKonqueror() ||
						$this->checkBrowserIcab() ||
						$this->checkBrowserOmniWeb() ||
						$this->checkBrowserPhoenix() ||
						$this->checkBrowserWebTv() ||
						$this->checkBrowserAmaya() ||
						$this->checkBrowserLynx() ||
						$this->checkBrowseriPhone() ||
						$this->checkBrowseriPod() ||
                        $this->checkBrowserBlackBerry() ||
						$this->checkBrowserW3CValidator() ||
						$this->checkBrowserMozilla() /* Mozilla is such an open standard that you must check it last */	
						);
		}

		/**
		 * Determine if the user is using a BlackBerry
		 * @return boolean True if the browser is the BlackBerry browser otherwise false
		 */
		function checkBrowserBlackBerry() {
			$retval = false;
			if( preg_match('/blackberry/i',$this->_agent) ) {
				$aresult = explode("/",stristr($this->_agent,"BlackBerry"));
				$aversion = explode(' ',$aresult[1]);
				$this->setVersion($aversion[0]);
				$this->_browser_name = $this->BROWSER_BLACKBERRY;
				$this->setMobile();
				$retval = true;
			}
			return $retval;
		}

		/**
		 * Determine if the user is using an AOL User Agent
		 * @return boolean True if the browser is from AOL otherwise false
		 */
		function checkForAol() {
			$retval = false;
			if( preg_match('/aol/i', $this->_agent) ) {
				$aversion = explode(' ',stristr($this->_agent, 'AOL'));
				$this->setAol(true);
				$this->setAolVersion(preg_replace('/[^0-9\.a-z]/i', '', $aversion[1]));
				$retval = true;
			}
			else {
				$this->setAol(false);
				$this->setAolVersion($this->VERSION_UNKNOWN);
				$retval = true;
			}
			return $retval;
		}
		
		/**
		 * Determine if the browser is the GoogleBot or not
		 * @return boolean True if the browser is the GoogletBot otherwise false
		 */
		function checkBrowserGoogleBot() {
			$retval = false;
			if( preg_match('/googlebot/i',$this->_agent) ) {
				$aresult = explode('/',stristr($this->_agent,'googlebot'));
				$aversion = explode(' ',$aresult[1]);
				$this->setVersion(str_replace(';','',$aversion[0]));
				$this->_browser_name = $this->BROWSER_GOOGLEBOT;
                $this->setRobot();
				$retval = true;
			}
			return $retval;
		}
				
		/**
		 * Determine if the browser is the W3C Validator or not
		 * @return boolean True if the browser is the W3C Validator otherwise false
		 */
		function checkBrowserW3CValidator() {
			$retval = false;
			if( preg_match('/W3C-checklink/i',$this->_agent) ) {
				$aresult = explode('/',stristr($this->_agent,'W3C-checklink'));
				$aversion = explode(' ',$aresult[1]);
				$this->setVersion($aversion[0]);
				$this->_browser_name = $this->BROWSER_W3CVALIDATOR;
				$retval = true;
			}
			return $retval;
		}
		
		/**
		 * Determine if the browser is the W3C Validator or not
		 * @return boolean True if the browser is the W3C Validator otherwise false
		 */
		function checkBrowserSlurp() {
			$retval = false;
			if( preg_match('/Slurp/i',$this->_agent) ) {
				$aresult = explode('/',stristr($this->_agent,'Slurp'));
				$aversion = explode(' ',$aresult[1]);
				$this->setVersion($aversion[0]);
				$this->_browser_name = $this->BROWSER_SLURP;
                $this->setRobot();
				$retval = true;
			}
			return $retval;
		}
		
		/**
		 * Determine if the browser is Internet Explorer or not
		 * @return boolean True if the browser is Internet Explorer otherwise false
		 */
		function checkBrowserInternetExplorer() {
			$retval = false;

			// Test for v1 - v1.5 IE
			if( preg_match('/microsoft internet explorer/i', $this->_agent) ) {
				$this->setBrowser($this->BROWSER_IE);
				$this->setVersion('1.0');
				$aresult = stristr($this->_agent, '/');
				if( preg_match('/308|425|426|474|0b1/i', $aresult) ) {
					$this->setVersion('1.5');
				}
				$retval = true;
			}
			// Test for versions > 1.5
			else if( preg_match('/msie/i',$this->_agent) && !preg_match('/opera/i',$this->_agent) ) {
				$aresult = explode(' ',stristr(str_replace(';','; ',$this->_agent),'msie'));
				$this->setBrowser( $this->BROWSER_IE );
				$this->setVersion(str_replace(array('(',')',';'),'',$aresult[1]));
				$retval = true;
			}
			// Test for Pocket IE
			else if( preg_match('/mspie/i',$this->_agent) || preg_match('/pocket/i', $this->_agent) ) {
				$aresult = explode(' ',stristr($this->_agent,'mspie'));
				$this->setPlatform( $this->PLATFORM_WINDOWS_CE );
				$this->setBrowser( $this->BROWSER_POCKET_IE );
				$this->setMobile();
				
				if( preg_match('/mspie/i', $this->_agent) ) {
					$this->setVersion($aresult[1]);
				}
				else {
					$aversion = explode('/',$this->_agent);
					$this->setVersion($aversion[1]);
				}
				$retval = true;
			}
			return $retval;
		}
		
		/**
		 * Determine if the browser is Opera or not
		 * @return boolean True if the browser is Opera otherwise false
		 */
		function checkBrowserOpera() {
			$retval = false;
			if( preg_match('/opera mini/i',$this->_agent) ) {
				$resultant = stristr($this->_agent, 'opera mini');
				if( preg_match('/\//',$resultant) ) {
					$aresult = explode('/',$resultant);
					$aversion = explode(' ',$aresult[1]); 
					$this->setVersion($aversion[0]);
					$this->_browser_name = $this->BROWSER_OPERA_MINI;
					$this->setMobile();
                    $retval = true;
				}
				else {
					$aversion = explode(' ',stristr($resultant,'opera mini'));
					$this->setVersion($aversion[1]);
					$this->_browser_name = $this->BROWSER_OPERA_MINI;
					$this->setMobile();
					$retval = true;
				}
			}
			else if( preg_match('/opera/i',$this->_agent) ) {
				$resultant = stristr($this->_agent, 'opera');
				if( preg_match('/Version\/(10.*)$/',$resultant,$matches) ) {
					$this->setVersion($matches[1]);
					$this->_browser_name = $this->BROWSER_OPERA;
					$retval = true;
				}
				else if( preg_match('/\//',$resultant) ) {
					$aresult = explode('/',$resultant);
					$aversion = explode(' ',$aresult[1]); 
					$this->setVersion($aversion[0]);
					$this->_browser_name = $this->BROWSER_OPERA;
					$retval = true;
				}
				else {
					$aversion = explode(' ',stristr($resultant,'opera'));
					$this->setVersion($aversion[1]);
					$this->_browser_name = $this->BROWSER_OPERA;
					$retval = true;
				}
			}
			return $retval;
		}
		
		/**
		 * Determine if the browser is WebTv or not
		 * @return boolean True if the browser is WebTv otherwise false
		 */
		function checkBrowserWebTv() {
			$retval = false;
			if( preg_match('/webtv/i',$this->_agent) ) {
				$aresult = explode('/',stristr($this->_agent,'webtv'));
				$aversion = explode(' ',$aresult[1]);
				$this->setVersion($aversion[0]);
				$this->_browser_name = $this->BROWSER_WEBTV;
				$retval = true;
			}
			return $retval;
		}
				
		/**
		 * Determine if the browser is NetPositive or not
		 * @return boolean True if the browser is NetPositive otherwise false
		 */
		function checkBrowserNetPositive() {
			$retval = false;
			if( preg_match('/NetPositive/i',$this->_agent) ) {
				$aresult = explode('/',stristr($this->_agent,'NetPositive'));
				$aversion = explode(' ',$aresult[1]);
				$this->setVersion(str_replace(array('(',')',';'),'',$aversion[0]));
				$this->_browser_name = $this->BROWSER_NETPOSITIVE;
				$this->_platform = $this->PLATFORM_BEOS;
				$retval = true;
			}
			return $retval;
		}
			
		/**
		 * Determine if the browser is Galeon or not
		 * @return boolean True if the browser is Galeon otherwise false
		 */
		function checkBrowserGaleon() {
			$retval = false;
			if( preg_match('/galeon/i',$this->_agent) ) {
				$aresult = explode(' ',stristr($this->_agent,'galeon'));
				$aversion = explode('/',$aresult[0]);
				$this->setVersion($aversion[1]);
				$this->setBrowser($this->BROWSER_GALEON);
				$retval = true;
			}
			return $retval;
		}
			
		/**
		 * Determine if the browser is Konqueror or not
		 * @return boolean True if the browser is Konqueror otherwise false
		 */
		function checkBrowserKonqueror() {
			$retval = false;
			if( preg_match('/Konqueror/i',$this->_agent) ) {
				$aresult = explode(' ',stristr($this->_agent,'Konqueror'));
				$aversion = explode('/',$aresult[0]);
				$this->setVersion($aversion[1]);
				$this->setBrowser($this->BROWSER_KONQUEROR);
				$retval = true;
			}
			return $retval;
		}
			
		/**
		 * Determine if the browser is iCab or not
		 * @return boolean True if the browser is iCab otherwise false
		 */
		function checkBrowserIcab() {
			$retval = false;
			if( preg_match('/icab/i',$this->_agent) ) {
				$aversion = explode(' ',stristr(str_replace('/',' ',$this->_agent),'icab'));
				$this->setVersion($aversion[1]);
				$this->setBrowser($this->BROWSER_ICAB);
				$retval = true;
			}
			return $retval;
		}
			
		/**
		 * Determine if the browser is OmniWeb or not
		 * @return boolean True if the browser is OmniWeb otherwise false
		 */
		function checkBrowserOmniWeb() {
			$retval = false;
			if( preg_match('/omniweb/i',$this->_agent) ) {
				$aresult = explode('/',stristr($this->_agent,'omniweb'));
				$aversion = explode(' ',$aresult[1]);
				$this->setVersion($aversion[0]);
				$this->setBrowser($this->BROWSER_OMNIWEB);
				$retval = true;
			}
			return $retval;
		}
			
		/**
		 * Determine if the browser is Phoenix or not
		 * @return boolean True if the browser is Phoenix otherwise false
		 */
		function checkBrowserPhoenix() {
			$retval = false;
			if( preg_match('/Phoenix/i',$this->_agent) ) {
				$aversion = explode('/',stristr($this->_agent,'Phoenix'));
				$this->setVersion($aversion[1]);
				$this->setBrowser($this->BROWSER_PHOENIX);
				$retval = true;
			}
			return $retval;
		}
		
		/**
		 * Determine if the browser is Firebird or not
		 * @return boolean True if the browser is Firebird otherwise false
		 */
		function checkBrowserFirebird() {
			$retval = false;
			if( preg_match('/Firebird/i',$this->_agent) ) {
				$aversion = explode('/',stristr($this->_agent,'Firebird'));
				$this->setVersion($aversion[1]);
				$this->setBrowser($this->BROWSER_FIREBIRD);
				$retval = true;
			}
			return $retval;
		}
		
		/**
		 * Determine if the browser is Netscape Navigator 9+ or not (http://browser.netscape.com/ - Official support ended on March 1st, 2008)
		 * @return boolean True if the browser is Netscape Navigator 9+ otherwise false
		 */
		function checkBrowserNetscapeNavigator9Plus() {
			$retval = false;
			if( preg_match('/Firefox/i',$this->_agent) && preg_match('/Navigator\/([^ ]*)/i',$this->_agent,$matches) ) {
				$this->setVersion($matches[1]);
				$this->setBrowser($this->BROWSER_NETSCAPE_NAVIGATOR);
				$retval = true;
			}
			return $retval;
		}
		
		/**
		 * Determine if the browser is Shiretoko or not (https://wiki.mozilla.org/Projects/shiretoko)
		 * @return boolean True if the browser is Shiretoko otherwise false
		 */
		function checkBrowserShiretoko() {
			$retval = false;
			if( preg_match('/Mozilla/i',$this->_agent) && preg_match('/Shiretoko\/([^ ]*)/i',$this->_agent,$matches) ) {
				$this->setVersion($matches[1]);
				$this->setBrowser($this->BROWSER_SHIRETOKO);
				$retval = true;
			}
			return $retval;
		}
		
		/**
		 * Determine if the browser is Ice Cat or not (http://en.wikipedia.org/wiki/GNU_IceCat)
		 * @return boolean True if the browser is Ice Cat otherwise false
		 */
		function checkBrowserIceCat() {
			$retval = false;
			if( preg_match('/Mozilla/i',$this->_agent) && preg_match('/IceCat\/([^ ]*)/i',$this->_agent,$matches) ) {
				$this->setVersion($matches[1]);
				$this->setBrowser($this->BROWSER_ICECAT);
				$retval = true;
			}
			return $retval;
		}
		
		/**
		 * Determine if the browser is Firefox or not
		 * @return boolean True if the browser is Firefox otherwise false
		 */
		function checkBrowserFirefox() {
			$retval = false;
			if( preg_match('/Firefox/i',$this->_agent) ) {
				$aresult = explode('/',stristr($this->_agent,'Firefox'));
				$aversion = explode(' ',$aresult[1]);
				$this->setVersion($aversion[0]);
				$this->setBrowser($this->BROWSER_FIREFOX);
				$retval = true;
			}
			return $retval;
		}
		
		/**
		 * Determine if the browser is Mozilla or not
		 * @return boolean True if the browser is Mozilla otherwise false
		 */
		function checkBrowserMozilla() {
			$retval = false;
			if( preg_match('/mozilla/i',$this->_agent) && preg_match('/rv:[0-9].[0-9][a-b]?/i',$this->_agent) && !preg_match('/netscape/i',$this->_agent)) {
				$aversion = explode(' ',stristr($this->_agent,'rv:'));
				preg_match('/rv:[0-9].[0-9][a-b]?/i',$this->_agent,$aversion);
				$this->setVersion(str_replace('rv:','',$aversion[0]));
				$this->setBrowser($this->BROWSER_MOZILLA);
				$retval = true;
			}
			else if( preg_match('/mozilla/i',$this->_agent) && preg_match('/rv:[0-9]\.[0-9]/i',$this->_agent) && !preg_match('/netscape/i',$this->_agent) ) {
				$aversion = explode('',stristr($this->_agent,'rv:'));
            	preg_match('/rv:[0-9]\.[0-9]\.[0-9]/i',$this->_agent,$aversion);
            	echo 
				$this->setVersion(str_replace('rv:','',$aversion[0]));
				$this->setBrowser($this->BROWSER_MOZILLA);
				$retval = true;
			}
			return $retval;
		}

		/**
		 * Determine if the browser is Lynx or not
		 * @return boolean True if the browser is Lynx otherwise false
		 */
		function checkBrowserLynx() {
			$retval = false;
			if( preg_match('/libwww/i',$this->_agent) && preg_match('/lynx/i', $this->_agent) ) {
				$aresult = explode('/',stristr($this->_agent,'Lynx'));
				$aversion = explode(' ',$aresult[1]);
				$this->setVersion($aversion[0]);
				$this->setBrowser($this->BROWSER_LYNX);
				$retval = true;
			}
			return $retval;
		}
			
		/**
		 * Determine if the browser is Amaya or not
		 * @return boolean True if the browser is Amaya otherwise false
		 */
		function checkBrowserAmaya() {
			$retval = false;
			if( preg_match('/libwww/i',$this->_agent) && preg_match('/amaya/i', $this->_agent) ) {
				$aresult = explode('/',stristr($this->_agent,'Amaya'));
				$aversion = explode(' ',$aresult[1]);
				$this->setVersion($aversion[0]);
				$this->setBrowser($this->BROWSER_AMAYA);
				$retval = true;
			}
			return $retval;
		}
			
		/**
		 * Determine if the browser is Chrome or not
		 * @return boolean True if the browser is Chrome otherwise false
		 */
		function checkBrowserChrome() {
			$retval = false;
			if( preg_match('/Chrome/i',$this->_agent) ) {
				$aresult = explode('/',stristr($this->_agent,'Chrome'));
				$aversion = explode(' ',$aresult[1]);
				$this->setVersion($aversion[0]);
				$this->setBrowser($this->BROWSER_CHROME);
				$retval = true;
			}
			return $retval;
		}		
		
		/**
		 * Determine if the browser is Safari or not
		 * @return boolean True if the browser is Safari otherwise false
		 */
		function checkBrowserSafari() {
			$retval = false;
			if( preg_match('/Safari/i',$this->_agent) && ! preg_match('/iPhone/i',$this->_agent) && ! preg_match('/iPod/i',$this->_agent) ) {
				$aresult = explode('/',stristr($this->_agent,'Version'));
				if( isset($aresult[1]) ) {
					$aversion = explode(' ',$aresult[1]);
					$this->setVersion($aversion[0]);
				}
				else {
					$this->setVersion($this->VERSION_UNKNOWN);
				}
				$this->setBrowser($this->BROWSER_SAFARI);
				$retval = true;
			}
			return $retval;
		}		
		
		/**
		 * Determine if the browser is iPhone or not
		 * @return boolean True if the browser is iPhone otherwise false
		 */
		function checkBrowseriPhone() {
			$retval = false;
			if( preg_match('/iPhone/i',$this->_agent) ) {
				$aresult = explode('/',stristr($this->_agent,'Version'));
				if( isset($aresult[1]) ) {
					$aversion = explode(' ',$aresult[1]);
					$this->setVersion($aversion[0]);
				}
				else {
					$this->setVersion($this->VERSION_UNKNOWN);
				}
				$this->setMobile();
				$this->setBrowser($this->BROWSER_IPHONE);
				$retval = true;
			}
			return $retval;
		}		

		/**
		 * Determine if the browser is iPod or not
		 * @return boolean True if the browser is iPod otherwise false
		 */
		function checkBrowseriPod() {
			$retval = false;
			if( preg_match('/iPod/i',$this->_agent) ) {
				$aresult = explode('/',stristr($this->_agent,'Version'));
				if( isset($aresult[1]) ) {
					$aversion = explode(' ',$aresult[1]);
					$this->setVersion($aversion[0]);
				}
				else {
					$this->setVersion($this->VERSION_UNKNOWN);
				}
				$this->setMobile();
				$this->setBrowser($this->BROWSER_IPOD);
				$retval = true;
			}
			return $retval;
		}		

		/**
		 * Determine if the browser is iPod or not
		 * @return boolean True if the browser is iPod otherwise false
		 */
		function checkBrowseriPad() {
			$retval = false;
			if( preg_match('/iPad/i',$this->_agent) ) {
				$aresult = explode('/',stristr($this->_agent,'Version'));
				if( isset($aresult[1]) ) {
					$aversion = explode(' ',$aresult[1]);
					$this->setVersion($aversion[0]);
				}
				else {
					$this->setVersion($this->VERSION_UNKNOWN);
				}
				$this->setMobile();
				$this->setBrowser($this->BROWSER_IPAD);
				$retval = true;
			}
			return $retval;
		}		

		/**
		 * Determine if the browser is Android or not
		 * @return boolean True if the browser is Android otherwise false
		 */
		function checkBrowserAndroid() {
			$retval = false;
			if( preg_match('/Android/i',$this->_agent) ) {
				$aresult = explode('/',stristr($this->_agent,'Version'));
				if( isset($aresult[1]) ) {
					$aversion = explode(' ',$aresult[1]);
					$this->setVersion($aversion[0]);
				}
				else {
					$this->setVersion($this->VERSION_UNKNOWN);
				}
				$this->setMobile();
				$this->setBrowser($this->BROWSER_ANDROID);
				$retval = true;
			}
			return $retval;
		}		

		/**
		 * Determine the user's platform
		 */
		function checkPlatform() {
			if( preg_match('/iPhone/i', $this->_agent) ) {
				$this->_platform = $this->PLATFORM_IPHONE;
			}
			else if( preg_match('/iPod/i', $this->_agent) ) {
				$this->_platform = $this->PLATFORM_IPOD;
			}
			else if( preg_match('/BlackBerry/i', $this->_agent) ) {
				$this->_platform = $this->PLATFORM_BLACKBERRY;
			}
			else if( preg_match('/win/i', $this->_agent) ) {
				$this->_platform = $this->PLATFORM_WINDOWS;
			}
			elseif( preg_match('/mac/i', $this->_agent) ) {
				$this->_platform = $this->PLATFORM_APPLE;
			}
			elseif( preg_match('/linux/i', $this->_agent) ) {
				$this->_platform = $this->PLATFORM_LINUX;
			}
			elseif( preg_match('/OS\/2/i', $this->_agent) ) {
				$this->_platform = $this->PLATFORM_OS2;
			}
			elseif( preg_match('/BeOS/i', $this->_agent) ) {
				$this->_platform = $this->PLATFORM_BEOS;
			}
		}
	}
?>
