<?php
    /**
     * 쿼리를 실행하거나 유틸을 가지고 있는 클래스
     * 
     * @class  cashbookModel
     * @author jedison (master@xgenesis.org)
     * @brief  cashbook 모듈의 Model class
     **/

    class cashbookModel extends cashbook {

        /**
         * @brief 초기화
         **/
        function init() {

        }
        
        /********************************************************************
         * modules
         * module_config
         * module_extra_vars
         * module_part_config
         * module_grants
         ********************************************************************/
    	/**
    	 * CashbookXE 모듈의 기본정보를 가져온다.
    	 * @return unknown_type
    	 */
        function getBasicModuleInfo() {
            static $module_info = null;
            if(is_null($module_info)) {
            	
                // module_config에서 cashbook으로 등록된 설정을 가져온다.
                $oModuleModel = &getModel('module');
                $module_info = $oModuleModel->getModuleConfig('cashbook');

                $skin_info->module_srl = $module_info->module_srl;
                $oModuleModel->syncSkinInfoToModuleInfo($skin_info);

                //module에서 등록된 기본설정을 가져온다.
                $dummy = $oModuleModel->getModuleInfoByMid($module_info->mid);                
                $module_info->is_default = $dummy->is_default;
                $module_info->module_srl = $dummy->module_srl;
                $module_info->browser_title = $dummy->browser_title;
                $module_info->layout_srl = $dummy->layout_srl;

                unset($module_info->grants);
            }
            return $module_info;
        }
        
    	/**
         * @brief 금전출납부 설정 목록 return
         **/
        function getConfigList($list_count=20, $page=1, $sort_index = 'module_srl') {
            $args->sort_index = $sort_index;
            $args->list_count = $list_count;
            $args->page = $page;
            $output = executeQueryArray('cashbook.getConfigList', $args);         
            return $output;
        }
        
    	/**
         * 현재 모듈에 설정된 권한 그룹을 가져옴
         * @return unknown_type
         */
        function getGrantGroup($module_srl){  	
        	$default_grant = array();
            $args->module_srl = $module_srl;
            $output = executeQueryArray('cashbook.getModuleGrants', $args);
            if($output->data) {
                foreach($output->data as $val) {
                    if($val->group_srl == 0) $default_grant[$val->name] = 'all';
                    else if($val->group_srl == -1) $default_grant[$val->name] = 'member';
                    else if($val->group_srl == -2) $default_grant[$val->name] = 'site';
                    else if($val->group_srl == -3) $default_grant[$val->name] = 'owner';
                    else {
                        $selected_group[$val->name][] = $val->group_srl;
                        $default_grant[$val->name] = 'group';
                    }
                }
            }
            $output->selected_group = $selected_group;
            $output->default_grant = $default_grant;
            
            return $output;
        }
	
   	 	/**
         * mid가 사용중인지 검사하기 위해 modules 를 뒤진다.
         * 
         * @param $list_count
         * @param $page
         * @param $sort_index
         * @return unknown_type
         */
        function getModule($args) {
            $output = executeQueryArray('cashbook.getModule', $args);
            return $output;
        }
        
    	/**
         * @brief 설정 1개
         **/
        function getConfig($args) {
            $output = executeQuery('cashbook.getConfig', $args);
            $part_config = unserialize($output->data->config);
            $config = $part_config[0];
            $config->site_srl = $output->data->site_srl;
            $config->module_srl = $output->data->module_srl;
            $config->member_srl = $output->data->member_srl;
            $config->browser_title = $output->data->browser_title;
            $config->mid = $output->data->mid;
            return $config;
        }
        
    	/**
    	 * @brief 설정 입력
    	 * 
    	 * @param $args
    	 * @return unknown_type
    	 */
        function insertConfig($args) {
            // 쿼리실행
            $output = executeQuery('cashbook.insertConfig', $args);
            return $output;
        }
        
    	/**
    	 * @brief 설정 수정
    	 * 
    	 * @param $args
    	 * @return unknown_type
    	 */
        function updateConfig($args) {       	
            // 쿼리실행
            $output = executeQuery('cashbook.updateConfig', $args);
            return $output;
        }
        
    	/**
         * @brief modules 입력
         * 
         * @param $args
         * @return unknown_type
         */
        function insertModule($args){
        	$output = $this->arrangeModuleInfo($args, $extra_vars);
            if(!$output->toBool()) return $output;

            // 이미 존재하는 모듈 이름인지 체크
            if(!$args->site_srl) $args->site_srl = 0;
            if($this->isIDExists($args->mid, $args->site_srl)) return new Object(-1, 'msg_module_name_exists');

            // begin transaction
            $oDB = &DB::getInstance();
            $oDB->begin();

            // 선택된 스킨정보에서 colorset을 구함
            $module_path = ModuleHandler::getModulePath($args->module);
            $skin_info = $this->loadSkinInfo($module_path, $args->skin);
            $skin_vars->colorset = $skin_info->colorset[0]->name;

            // 변수 정리후 query 실행
            if(!$args->module_srl) $args->module_srl = getNextSequence();

            // 모듈 등록
            $output = executeQuery('module.insertModule', $args);
            if(!$output->toBool()) {
                $oDB->rollback();
                return $output;
            }

            // commit
            $oDB->commit();

            $output->add('module_srl',$args->module_srl);
            return $output;
        }
        
    	/**
         * @brief 특정 mid의 모듈 설정 정보 저장
         * mid의 모듈 의존적인 설정을 관리
         **/
        function insertModulePartConfig($module, $module_srl, $config) {
        	$oModuleController = &getController('module');    
			$output = $oModuleController->insertModulePartConfig($module, $module_srl, $config);
            return $output;
        }
        
    	/**
         * @brief 모듈 정보 정리
         **/
        function arrangeModuleInfo(&$args, &$extra_vars) {
            // 불필요한 내용 제거
            unset($args->body);
            unset($args->act);
            unset($args->page);

            // mid값 검사
            if(!preg_match("/^[a-z][a-z0-9_]+$/i", $args->mid)) return new Object(-1, 'msg_limit_mid');

            // 변수를 검사 (modules의 기본 변수와 그렇지 않은 변수로 분리)
            $extra_vars = clone($args);
            unset($extra_vars->filter);
            unset($extra_vars->module_srl);
            unset($extra_vars->module);
            unset($extra_vars->module_category_srl);
            unset($extra_vars->layout_srl);
            unset($extra_vars->menu_srl);
            unset($extra_vars->site_srl);
            unset($extra_vars->mid);
            unset($extra_vars->skin);
            unset($extra_vars->browser_title);
            unset($extra_vars->description);
            unset($extra_vars->is_default);
            unset($extra_vars->content);
            unset($extra_vars->open_rss);
            unset($extra_vars->header_text);
            unset($extra_vars->footer_text);
            $args = delObjectVars($args, $extra_vars);

            return new Object();
        }
        
    	/**
         * @brief 특정 위치의 특정 스킨의 정보를 구해옴
         **/
        function loadSkinInfo($path, $skin, $dir = 'skins') {
            $oModuleModel = &getModel('module');    
			$skin_info = $oModuleModel->loadSkinInfo($path,$skin);
            return $skin_info;
        }
        
    	/**
         * @brief mid, vid 사용할 수 있는지 검사
         **/
        function isIDExists($id, $site_srl = 0) {
        	$oModuleModel = &getModel('module');    
			$output = $oModuleModel->isIDExists($id, $site_srl);
            return $output;
        }
        
    	/**
         * @brief 특정 모듈에 권한 등록
         **/
        function insertModuleGrants($module_srl, $obj) {
        	$this->deleteModuleGrants($module_srl);

            if(!$obj || !count($obj)) return;

            foreach($obj as $name => $val) {
                if(!$val || !count($val)) continue;
                foreach($val as $group_srl) {
                    $args = null;
                    $args->module_srl = $module_srl;
                    $args->name = $name;
                    $args->group_srl = trim($group_srl);                   
                    if(!$args->name || !$args->group_srl) continue;
					$output = executeQuery('module.insertModuleGrant', $args);
                }
            }
            return $output;
        }
        
    	/**
         * @brief 특정 모듈의 권한 제거
         **/
        function deleteModuleGrants($module_srl) {
            $oModuleController = &getController('module'); 
            $output = $oModuleController->deleteModuleGrants($module_srl);
            return $output;
        }
        
    	/**
         * @brief 특정 모듈에 관리자 아이디 지정
         **/
        function insertAdminId($module_srl, $admin_id) {
            $oMemberModel = &getModel('member');    
            $member_info = $oMemberModel->getMemberInfoByUserID($admin_id);        
            if(!$member_info->member_srl) return;
            $args->module_srl = $module_srl;
            $args->member_srl = $member_info->member_srl;
            return executeQuery('module.insertAdminId', $args);
        }
        
    	/**
         * @brief 주어진 곳의 스킨 목록을 구함
         * 스킨과 skin.xml 파일을 분석 정리한 결과를 return
         **/
        function getSkins($path, $dir = 'skins') {
        	$oModuleModel = &getModel('module');    
            $skin_list = $oModuleModel->getSkins($path);
            return $skin_list;
        }
        /********************************************************************
         * cashbook_account
         ********************************************************************/
    	
    	/**
         * @brief 계정 목록 전체
         * 
         * @param $list_count
         * @param $page
         * @param $sort_index
         * @return unknown_type
         */
        function getAccountListAll($args) {
            $output = executeQueryArray('cashbook.getAccountListAll', $args);           
            return $output;
        }

    	/**
         * @brief 계정 목록
         **/
        function getAccountList($args) {
            $output = executeQueryArray('cashbook.getAccountList', $args);
            return $output;
        }
        
    	/**
         * @brief 계정 1개
         * 
         * @return unknown_type
         */
        function getAccount($args) {
            $output = executeQuery('cashbook.getAccount', $args);        
            return $output->data;
        }
                
    	/**
    	 * @brief 계정 입력
    	 * 
    	 * @param $args
    	 * @return unknown_type
    	 */
        function insertAccount($args) {
            // 쿼리실행
            $output = executeQuery('cashbook.insertAccount', $args);
            return $output;
        }
        
    	/**
    	 * @brief 계정 수정
    	 * 
    	 * @param $args
    	 * @return unknown_type
    	 */
        function updateAccount($args) {       	
            $output = executeQuery('cashbook.updateAccount', $args);
            return $output;
        }
        
    	/**
    	 * @brief 계정 삭제
    	 * 
    	 * @param $args
    	 * @return unknown_type
    	 */
        function deleteAccount($args) {       	
            $output = executeQuery('cashbook.deleteAccount', $args);
            return $output;
        }
        
    	/**
         * 계정전체삭제
         * 
         * @param $args
         * @return unknown_type
         */
    	function deleteAccountAll($args) {
            $output = executeQuery('cashbook.deleteAccountAll', $args);   
            return $output;
        }
        
        /********************************************************************
         * cashbook_category
         * cashbook_category_item
         ********************************************************************/
    	/**
         * @brief 분류 목록 전체
         * @return unknown_type
         */
        function getCategoryListAll($args) {        	
            // 쿼리 실행
            $output = executeQuery('cashbook.getCategoryListAll', $args);            
            return $output;
        }
    	
    	/**
         * @brief 분류 목록
         * 
         * @return unknown_type
         */
        function getCategoryList($args) {
            $output = executeQuery('cashbook.getCategoryList', $args);         
            return $output;
        }
        
    	/**
         * @brief 분류 1개
         * 
         * DB 에 생성된 한개의 메뉴 정보를 구함
         * 생성된 메뉴의 DB정보+XML정보를 return
         * @return unknown_type
         */
        function getCategory($category_srl) {
            // 일단 DB에서 정보를 가져옴
            $args->category_srl = $category_srl;
            $output = executeQuery('cashbook.getCategory', $args);
            if(!$output->data) return;

            $category_info = $output->data;
            $category_info->xml_file = sprintf('./files/cache/cashbook/%s.xml.php',$category_srl);
            $category_info->php_file = sprintf('./files/cache/cashbook/%s.php',$category_srl);
            return $category_info;
        }
        
        /**
         * 분류에 등록된 자식의 수를 가져온다.
         * @param $args
         * @return unknown_type
         */
    	function getCategoryChildCount($args) {
            // 쿼리실행
            $output =  executeQuery('cashbook.getCategoryChildCount', $args);
            return $output;
        }
        
    	/**
    	 * @brief 분류 입력
    	 * 
    	 * @param $args
    	 * @return unknown_type
    	 */
        function insertCategory($args) {
            // 쿼리실행
            $output = executeQuery('cashbook.insertCategory', $args);
            return $output;
        }
        
    	/**
         * @brief 분류 수정
         * 
         * @param $args
         * @return unknown_type
         */
    	function updateCategory($args) {
            // 쿼리실행
            $output = executeQuery('cashbook.updateCategory', $args);

            if(!$output->toBool()) {
            	return $output;
            }
			
			$output->add('category_srl', $args->category_srl);
             
            return $output;
        }
        
    	/**
         * @brief 분류항목 목록 전체
         * 
         * 수입/지출 등록시 항목을 가져온다.
         * 
         * @param $args
         * @return unknown_type
         */
        function getCategoryItemListAll($args){       	
            $output =  executeQuery("cashbook.getCategoryItemListAll", $args);       
            return $output;
        }
        
    	/**
         * @brief 분류항목 1개
         * 
         * 특정 cashbook_category_srl의 아이템 정보를 return
         * 이 정보중에 group_srls의 경우는 , 로 연결되어 들어가며 사용시에는 explode를 통해 array로 변환 시킴
         **/
        function getCategoryItem($category_item_srl) {
            // category_item_srl이 있으면 해당 분류의 정보를 가져온다
            $args->category_item_srl = $category_item_srl;
            $output = executeQuery('cashbook.getCategoryItem', $args);

            $node = $output->data;
            if($node->group_srls) $node->group_srls = explode(',',$node->group_srls);
            else $node->group_srls = array();

            $tmp_name = unserialize($node->name);
            if($tmp_name && count($tmp_name) ) {
                $selected_lang = array();
                $rand_name = $tmp_name[Context::getLangType()];
                if(!$rand_name) $rand_name = array_shift($tmp_name);
                $node->name = $rand_name;
            }
            return $node;
        }
        
    	/**
         * @brief 특정 cashbook_category_srl의 아이템 정보를 return
         * 이 정보중에 group_srls의 경우는 , 로 연결되어 들어가며 사용시에는 explode를 통해 array로 변환 시킴
         **/
        function getCategoryItemByCategoryItemSrl($args) {
            $output = executeQuery('cashbook.getCategoryItemByCategoryItemSrl', $args);
            $node = $output->data;
            if($node->group_srls) $node->group_srls = explode(',',$node->group_srls);
            else $node->group_srls = array();

            $tmp_name = unserialize($node->name);
            if($tmp_name && count($tmp_name) ) {
                $selected_lang = array();
                $rand_name = $tmp_name[Context::getLangType()];
                if(!$rand_name) $rand_name = array_shift($tmp_name);
                $node->name = $rand_name;
            }
            return $node;
        }
        
            
    	/**
         * @brief 분류항목 입력
         * 
         * @param $args
         * @return unknown_type
         */
    	function insertCategoryItem($args) {
            // 쿼리실행
            $output = executeQuery('cashbook.insertCategoryItem', $args);
            if(!$output->toBool()) return $output;
			
            $output->add('category_item_srl', $args->category_item_srl);
            $output->add('category_srl', $args->category_srl);
            
            return $output;
        }

    	/**
         * @brief 분류항목 수정
         * 
         * @param $args
         * @return unknown_type
         */
    	function updateCategoryItem($args) {
            // 쿼리실행
            $output = executeQuery('cashbook.updateCategoryItem', $args);
            if(!$output->toBool()) return $output;

            return $output;
        }
        
    	/**
         * @brief 분류항목 삭제
         * @param $args
         * @return unknown_type
         */
    	function deleteCategoryItem($args) {
            // 쿼리실행
            $output =  executeQuery("cashbook.deleteCategoryItem", $args);
            // 결과가 없거나 오류 발생시 그냥 return
            if(!$output->toBool()) return $output;

            return $output;
        }
        
    	/**
         * 가장 큰 listorder를 가져온다.
         * 
         * @param $args
         * @return unknown_type
         */
    	function getCategoryMaxListorder($args) {
            // 쿼리실행
            $output = executeQuery('cashbook.getCategoryMaxListorder', $args);
            // 결과가 없거나 오류 발생시 그냥 return
            if(!$output->toBool()) return $output;

            return $output;
        }
        
    	/**
         * listorder를 업데이트한다.
         * 
         * @param $args
         * @return unknown_type
         */
    	function updateCategoryItemListorder($args) {
            // 쿼리실행
            $output = executeQuery('cashbook.updateCategoryItemListorder', $args);
            // 결과가 없거나 오류 발생시 그냥 return
            if(!$output->toBool()) return $output;

            return $output;
        }
        
		/**
		 * node를 수정한다.
		 * @param $args
		 * @return unknown_type
		 */
    	function updateCategoryItemNode($args) {
            // 쿼리실행
            $output =  executeQuery('cashbook.updateCategoryItemNode', $args);
            // 결과가 없거나 오류 발생시 그냥 return
            if(!$output->toBool()) return $output;

            return $output;
        }
        
    	/**
         * 분류 전체 삭제
         * @param $args
         * @return unknown_type
         */
    	function deleteCategoryAll($args) {
            $output =  executeQuery("cashbook.deleteCategoryAll", $args);
            return $output;
        }
        
    	/**
         * 분류항목 전체삭제
         * @param $args
         * @return unknown_type
         */
    	function deleteCategoryItemAll($args) {
            $output =  executeQuery("cashbook.deleteCategoryItemAll", $args);
            return $output;
        }
        
        
        /********************************************************************
         * cashbook_customer
         ********************************************************************/
    	/**
         * @brief 거래처 목록 전체
         * 
         * @param $list_count
         * @param $page
         * @param $sort_index
         * @return unknown_type
         */
        function getCustomerListAll($args) {
            $output = executeQueryArray('cashbook.getCustomerListAll', $args);
            return $output;
        }
        
        /**
         * @brief 거래처 목록
         * 
         * @param $list_count
         * @param $page
         * @param $sort_index
         * @return unknown_type
         */
        function getCustomerList($args) {
            $output = executeQueryArray('cashbook.getCustomerList', $args);
            return $output;
        }
        
    	/**
         * @brief 거래처 1개
         * 
         * @param $list_count
         * @param $page
         * @param $sort_index
         * @return unknown_type
         */
        function getCustomer($args) {
            $output = executeQueryArray('cashbook.getCustomer', $args);
            return $output->data[0];
        }
        
    	/**
         * @brief 거래처 입력
         * 
         * @param $args
         * @return unknown_type
         */
    	function insertCustomer($args) {
            $output = executeQuery('cashbook.insertCustomer', $args);
            return $output;
        }
        
    	/**
         * @brief 거래처 수정
         * 
         * @param $args
         * @return unknown_type
         */
    	function updateCustomer($args) {
            $output = executeQuery('cashbook.updateCustomer', $args);          
            return $output;
        }
        
    	/**
         * @brief 거래처 삭제
         * 
         * @param $args
         * @return unknown_type
         */
    	function deleteCustomer($args) {
            $output = executeQuery('cashbook.deleteCustomer', $args);          
            return $output;
        }
        
    	/**
         * 거래처전체삭제
         * 
         * @param $args
         * @return unknown_type
         */
    	function deleteCustomerAll($args) {
            $output = executeQuery('cashbook.deleteCustomerAll', $args);
            return $output;
        }
        
        /********************************************************************
         * cashbook_deal 
         ********************************************************************/
    	
    	/**
         * @brief 거래 1개 가져오기
         **/
        function getDeal($args) {
            $output = executeQuery('cashbook.getDeal', $args);           
            return $output->data;
        }
        
    	/**
         * @brief 거래 목록 전체
         * 
         * @param $list_count
         * @param $page
         * @param $sort_index
         * @return unknown_type
         */
        function getDealListAll($args) {
            $output = executeQueryArray('cashbook.getDealListAll', $args);
            return $output;
        }
        
    	/**
         * @brief 기간별 거래 목록
         * 
         * @param $list_count
         * @param $page
         * @param $sort_index
         * @return unknown_type
         */
        function getDealListByPeriod($args) {
            $output = executeQueryArray('cashbook.getDealListByPeriod', $args);            
            return $output->data;
        }
        
    	/**
         * @brief 수입 목록
         * 
         * @param $list_count
         * @param $page
         * @param $sort_index
         * @return unknown_type
         */
        function getIncomeList($args) {
            $output = executeQueryArray('cashbook.getIncomeList', $args);
            return $output;
        }
        
    	/**
         * @brief 지출 목록
         * 
         * @param $list_count
         * @param $page
         * @param $sort_index
         * @return unknown_type
         */
        function getOutgoList($args) {
            $output = executeQueryArray('cashbook.getOutgoList', $args);
            return $output;
        }
        
    	/**
         * @brief 이체 목록
         * 
         * @param $list_count
         * @param $page
         * @param $sort_index
         * @return unknown_type
         */
        function getTransferList($args) {
            $output = executeQueryArray('cashbook.getTransferList', $args);
            return $output;
        }
        
    	/**
         * @brief 거래(수입, 지출, 이체) 1개
         **/
        function getDealByAccount($args) {
            $output = executeQuery('cashbook.getDealByAccount', $args);
			return $output;
        }
        
    	/**
         * @brief 거래 등록
         * 
         * @param $args
         * @return unknown_type
         */
    	function insertDeal($args) {
            $output = executeQuery('cashbook.insertDeal', $args);           
            return $output;
        }
        
    	/**
         * @brief 거래 수정
         * 
         * @param $args
         * @return unknown_type
         */
    	function updateDeal($args) {
            $output = executeQuery('cashbook.updateDeal', $args);         
            return $output;
        }

    	/**
         * @brief 거래 삭제
         * 
         * @param $args
         * @return unknown_type
         */
    	function deleteDeal($args) {
            $output = executeQuery('cashbook.deleteDeal', $args);
			return $output;
        }
        
    	/**
         * 월간 수입 목록
         * 
         * @param $list_count
         * @param $page
         * @param $sort_index
         * @return unknown_type
         */
        function getIncomeMonthlyList($args) {	
            $output = executeQueryArray('cashbook.getIncomeMonthlyList', $args);
            return $output;
        }
        
    	/**
         * 월간 지출 목록
         * 
         * @param $list_count
         * @param $page
         * @param $sort_index
         * @return unknown_type
         */
        function getOutgoMonthlyList($args) { 
            $output = executeQueryArray('cashbook.getOutgoMonthlyList', $args);
            return $output;
        }
        
    	/**
         * 월간 이체 목록
         * 
         * @param $list_count
         * @param $page
         * @param $sort_index
         * @return unknown_type
         */
        function getTransferMonthlyList($args) {    
            $output = executeQueryArray('cashbook.getTransferMonthlyList', $args);
            return $output;
        }
        
    	/**
         * 기간에 따른 수입 총합
         * 
         * @param $list_count
         * @param $page
         * @param $sort_index
         * @return unknown_type
         */
        function getIncomePeriodSum($args) {
            $output = executeQueryArray('cashbook.getIncomePeriodSum', $args);       
            return $output->data[0];
        }
        
   		/**
         * 기간에 따른 지출 총합
         * 
         * @param $list_count
         * @param $page
         * @param $sort_index
         * @return unknown_type
         */
        function getOutgoPeriodSum($args) {
            $output = executeQueryArray('cashbook.getOutgoPeriodSum', $args);
            return $output->data[0];
        }
		
    	/**
         * 수입거래전체삭제
         * 
         * @param $args
         * @return unknown_type
         */
    	function deleteDealIncomeAll($args) {
            $output = executeQuery('cashbook.deleteDealIncomeAll', $args);
            return $output;
        }
        
    	/**
         * 지출거래전체삭제
         * 
         * @param $args
         * @return unknown_type
         */
    	function deleteDealOutgoAll($args) {
            $output = executeQuery('cashbook.deleteDealOutgoAll', $args);
            return $output;
        }
        
    	/**
         * 이체거래전체삭제
         * 
         * @param $args
         * @return unknown_type
         */
    	function deleteDealTransferAll($args) {
            $output = executeQuery('cashbook.deleteDealTransferAll', $args);
            return $output;
        }
        
        /********************************************************************
         * cashbook_tag 
         ********************************************************************/
		
    	/**
         * @brief 태그  목록 전체
         * 
         * @param $list_count
         * @param $page
         * @param $sort_index
         * @return unknown_type
         */
        function getTagListAll($args) {
            $output = executeQueryArray('cashbook.getTagListAll', $args);
            return $output;
        }
        
        /**
         * 
         * @param $args
         * @return unknown_type
         */
        function getTag($args) {
            $output = executeQueryArray('cashbook.getTag', $args);
            return $output->data[0];
        }
        
    	/**
         * 
         * @param $args
         * @return unknown_type
         */
        function insertTag($args) {
            $output = executeQueryArray('cashbook.insertTag', $args);
            return $output;
        }
        
    	/**
         * 
         * @param $args
         * @return unknown_type
         */
        function insertTagDeal($args) {
            $output = executeQueryArray('cashbook.insertTagDeal', $args);
            return $output;
        }
        
        
        
    	
        
    	
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        

                
    	
        
    	
        
        
        
    	
        
    	
        
    	
        
    	/********************************************************************
         * 유틸
         ********************************************************************/
    	
    	/**
         * @brief 권한검사
         * 
         * 권한이 있는지 검사한다.
         * @return unknown_type
         */
        function hasGrant($grant_name, $mid){
        	// 객체가져오기
        	$oModuleModel = &getModel('module');
			$oMemberModel = &getModel('member');
			
        	// 1.모듈정보를 가져온다.
        	if($mid==null){
        		$this->module_info = $oModuleModel->getModuleConfig("cashbook");
        	} else {        		
        		$this->module_info = $oModuleModel->getModuleInfoByMid($mid);
        	}

        	// 2.module_grants에 저장된 모듈의 권한을 가져온다.
        	$output = $this->getModuleGrants($this->module_info);
        	$grant_exists = array();
        	if($output->data) {
				// 권한이 부여된 그룹을 가져온다.
				foreach($output->data as $key=>$val) {
					if($val->name==$grant_name){
						$grant_exists[$key] = $val->group_srl;
					}
                }
        	}
        	// 비로그인 회원 모두
        	$hasGrant = in_array("0", $grant_exists);
        	$member_info = Context::get('logged_info'); 
        	
        	// 로그인 회원만
        	if(!$hasGrant) {
        		$tempGrant = in_array("-1", $grant_exists);
        		if($tempGrant && $member_info!=null){
        			$hasGrant = true;
        		}
        	}
        	// 사이트 가입한 회원만
        	if(!$hasGrant) {
        		$tempGrant = in_array("-2", $grant_exists);
        	}
        	// 소유자만
        	if(!$hasGrant) {
        		$tempGrant = in_array("-3", $grant_exists);
	            $site_module_info = Context::get('site_module_info');
	            $args->site_srl = (int)$site_module_info->site_srl;       
				$args->mid = $this->module_info->mid;
				$output = $this->getConfig($args);
        		if($tempGrant && $output->member_srl == $member_info->member_srl && $member_info!=null){
        			$hasGrant = true;
        		}
        	}
        	// 특정 그룹 대상일 경우
        	if(!$hasGrant){
	            $member_group_list = $this->getMemberGroupList($member_info);	            
	            foreach($member_group_list->data as $k=>$v) {
	            	$hasGrant = in_array($v->group_srl, $grant_exists);
	            }
        	}
        	
        	return $hasGrant;
        }
        
    	/**
         * @brief 분류  기본값 입력
         * 
         * @return unknown_type
         */
        function insertCategoryDefault($args, $title){
			$args->category_srl = getNextSequence();
			$args->title= $title;
			$args->allow_delete= "N";
			$args->listorder = $args->category_srl * -1;
            $output = $this->insertCategory($args);
            $output->category_srl = $args->category_srl;
            
            return $output;
        }
        
        /**
         * c@brief 분류항목 기본값 입력
         * 
         * @param $args
         * @param $parent_srl
         * @param $categor_srl
         * @param $title
         * @return unknown_type
         */
    	function insertCategoryItemDefault($args, $parent_srl, $categor_srl, $title){
            $args->category_item_srl = getNextSequence();
            $args->parent_srl = $parent_srl;
            $args->category_srl = $categor_srl;
			$args->category_title = $title;
			$args->listorder = $args->category_item_srl * -1;
            $output = $this->insertCategoryItem($args);
            $output->category_item_srl = $args->category_item_srl;
            return $output;
        }
		
        /********************************************************************
         * Flex (CashbookChart)
         ********************************************************************/
        
    	/**
         * 거래그래프 - 수입
         * @param $args
         * @return unknown_type
         */
        function getCashbookChartDealIncomeList($args){
        	$output = executeQueryArray('cashbook.getCashbookChartDealIncomeList', $args);
        	return $output;
        }
        
        /**
         * 거래그래프 - 지출
         * @param $args
         * @return unknown_type
         */
    	function getCashbookChartDealOutgoList($args){
        	$output = executeQueryArray('cashbook.getCashbookChartDealOutgoList', $args);
        	return $output;
        }
        
    	/**
         * 수입항목별그래프
         * @param $args
         * @return unknown_type
         */
    	function getCashbookChartIncomeItemList($args){
        	$output = executeQueryArray('cashbook.getCashbookChartIncomeItemList', $args);
        	return $output;
        }
        
    	/**
         * 지출항목별그래프
         * @param $args
         * @return unknown_type
         */
    	function getCashbookChartOutgoItemList($args){
        	$output = executeQueryArray('cashbook.getCashbookChartOutgoItemList', $args);
        	return $output;
        }
        
    	/**
         * 거래처별그래프 - 수입
         * @param $args
         * @return unknown_type
         */
    	function getCashbookChartCustomerIncomeList($args){ 
        	$output = executeQueryArray('cashbook.getCashbookChartCustomerIncomeList', $args);      	
        	return $output;
        }
        
    	/**
         * 거래처별그래프 - 지출
         * @param $args
         * @return unknown_type
         */
    	function getCashbookChartCustomerOutgoList($args){ 
        	$output = executeQueryArray('cashbook.getCashbookChartCustomerOutgoList', $args);
        	return $output;
        }
        
    	/**
         * 분류별그래프 - 수입
         * @param $args
         * @return unknown_type
         */
    	function getCashbookChartCategoryIncomeList($args){ 
        	$output = executeQueryArray('cashbook.getCashbookChartCategoryIncomeList', $args);      	
        	return $output;
        }
        
    	/**
         * 분류별그래프- 지출
         * @param $args
         * @return unknown_type
         */
    	function getCashbookChartCategoryOutgoList($args){ 
        	$output = executeQueryArray('cashbook.getCashbookChartCategoryOutgoList', $args);
        	return $output;
        }
        
    	/**
         * 태그별그래프 - 수입
         * @param $args
         * @return unknown_type
         */
    	function getCashbookChartTagIncomeList($args){ 
        	$output = executeQueryArray('cashbook.getCashbookChartTagIncomeList', $args);
        	return $output;
        }
        
    	/**
         * 태그별그래프 - 지출
         * @param $args
         * @return unknown_type
         */
    	function getCashbookChartTagOutgoList($args){ 
        	$output = executeQueryArray('cashbook.getCashbookChartTagOutgoList', $args);
        	return $output;
        }
        
        
        
        
    	/********************************************************************
         * 기본값 설정
         ********************************************************************/
            
    	/**
         * 초기값으로 설정하기
         * 초기값 설정대상: cashbook_account
         * 
         * @param $args
         * @return unknown_type
         */
        function setDefaultAccountData($args){     	
            // 기본값 입력하기 - cashbook_account 테이블
            // 입력할 정보 설정
            $args->account_srl = getNextSequence();		// 일련번호

			$args->kind = 1;	// Cash
			// 학생
			if(
				$args->family_type =='FAMILY_TYPE_01' || 
				$args->family_type =='FAMILY_TYPE_02' ||
				$args->family_type =='FAMILY_TYPE_03' ||
				$args->family_type =='FAMILY_TYPE_04' ||
				$args->family_type =='FAMILY_TYPE_05'
			){
				$args->title = "내지갑";
			// 직장인
			} else if(
				$args->family_type =='FAMILY_TYPE_11' ||
				$args->family_type =='FAMILY_TYPE_12'
			){
				$args->title = "내지갑";
			// 모임
			} else if(
				$args->family_type =='FAMILY_TYPE_21'){
				$args->title = "모임통장";
			// 회사
			} else if($args->family_type =='FAMILY_TYPE_31'){
				$args->title = "가계통장";
			// 회사
			} else if($args->family_type =='FAMILY_TYPE_32'){
				$args->title = "법인통장";
			}

            $tempYear = date("Y");
            $tempMonth = date("n");
            if($tempMonth < 10){
            	$tempMonth = "0".$tempMonth;
            }
            $tempDay = date("j");
        	if($tempDay < 10){
            	$tempDay = "0".$tempDay;
            }
            $args->day = $tempYear.$tempMonth.$tempDay;
            
            $args->amount = "0";
			$args->total_asset_yn = "Y";
            // cashbook_account 테이블에 입력
            $output = $this->insertAccount($args);
            
			return $output->account_srl;
        }

    	/**
         * @brief 수입항목 기본값 입력
         * 
         * @param $args
         * @return unknown_type
         */
        function setDefaultIncomeData($args){        	
			$output = $this->insertCategoryDefault($args,"수입항목");		
			$output1 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"근로소득");
			$this->insertCategoryItemDefault($args,$output1->category_item_srl,$output->category_srl,"기본급");
			$this->insertCategoryItemDefault($args,$output1->category_item_srl,$output->category_srl,"잔업수당");
			$this->insertCategoryItemDefault($args,$output1->category_item_srl,$output->category_srl,"연/월차수당");
			$this->insertCategoryItemDefault($args,$output1->category_item_srl,$output->category_srl,"설날상여금");
			$this->insertCategoryItemDefault($args,$output1->category_item_srl,$output->category_srl,"휴가상여금");
			$this->insertCategoryItemDefault($args,$output1->category_item_srl,$output->category_srl,"추석상여금");
			$this->insertCategoryItemDefault($args,$output1->category_item_srl,$output->category_srl,"연말상여금");
			$this->insertCategoryItemDefault($args,$output1->category_item_srl,$output->category_srl,"퇴직금");
			$this->insertCategoryItemDefault($args,$output1->category_item_srl,$output->category_srl,"기타");
			$output2 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"사업소득");
			$this->insertCategoryItemDefault($args,$output2->category_item_srl,$output->category_srl,"매출");
			$this->insertCategoryItemDefault($args,$output2->category_item_srl,$output->category_srl,"기타");
			$output3 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"임대소득");
			$this->insertCategoryItemDefault($args,$output3->category_item_srl,$output->category_srl,"전세보증금");
			$this->insertCategoryItemDefault($args,$output3->category_item_srl,$output->category_srl,"월세보증금");
			$this->insertCategoryItemDefault($args,$output3->category_item_srl,$output->category_srl,"월세");
			$this->insertCategoryItemDefault($args,$output3->category_item_srl,$output->category_srl,"기타");
			$output4 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"금융소득");
			$this->insertCategoryItemDefault($args,$output4->category_item_srl,$output->category_srl,"주식배당금");
			$this->insertCategoryItemDefault($args,$output4->category_item_srl,$output->category_srl,"채권배당금");
			$this->insertCategoryItemDefault($args,$output4->category_item_srl,$output->category_srl,"예금이자");
			$this->insertCategoryItemDefault($args,$output4->category_item_srl,$output->category_srl,"적금이자");
			$this->insertCategoryItemDefault($args,$output4->category_item_srl,$output->category_srl,"기타");
			$output5 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"증여소득");
			$this->insertCategoryItemDefault($args,$output5->category_item_srl,$output->category_srl,"부모");
			$this->insertCategoryItemDefault($args,$output5->category_item_srl,$output->category_srl,"형제");
			$this->insertCategoryItemDefault($args,$output5->category_item_srl,$output->category_srl,"기타");
			$output6 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"기타소득");
			$this->insertCategoryItemDefault($args,$output6->category_item_srl,$output->category_srl,"기타");

            return $output;
        }
        
    	/**
         * 지출항목 기본값 입력
         * 
         * @param $args
         * @return unknown_type
         */
        function setDefaultOutgoData($args){
			$output = $this->insertCategoryDefault($args,"지출항목");
			$output1 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"식비");
			$this->insertCategoryItemDefault($args,$output1->category_item_srl,$output->category_srl,"주식");
			$this->insertCategoryItemDefault($args,$output1->category_item_srl,$output->category_srl,"부식");
			$this->insertCategoryItemDefault($args,$output1->category_item_srl,$output->category_srl,"외식");
			$this->insertCategoryItemDefault($args,$output1->category_item_srl,$output->category_srl,"과일");
			$this->insertCategoryItemDefault($args,$output1->category_item_srl,$output->category_srl,"과자");
			$this->insertCategoryItemDefault($args,$output1->category_item_srl,$output->category_srl,"음료");
			$this->insertCategoryItemDefault($args,$output1->category_item_srl,$output->category_srl,"주류");
			$this->insertCategoryItemDefault($args,$output1->category_item_srl,$output->category_srl,"차류");
			$this->insertCategoryItemDefault($args,$output1->category_item_srl,$output->category_srl,"기타");
			$output2 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"주거비");
			$this->insertCategoryItemDefault($args,$output2->category_item_srl,$output->category_srl,"전기요금");
			$this->insertCategoryItemDefault($args,$output2->category_item_srl,$output->category_srl,"수도요금");
			$this->insertCategoryItemDefault($args,$output2->category_item_srl,$output->category_srl,"가스요금");
			$this->insertCategoryItemDefault($args,$output2->category_item_srl,$output->category_srl,"관리비");
			$this->insertCategoryItemDefault($args,$output2->category_item_srl,$output->category_srl,"월세");
		    $this->insertCategoryItemDefault($args,$output2->category_item_srl,$output->category_srl,"기타");
			$output3 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"통신비");
			$this->insertCategoryItemDefault($args,$output3->category_item_srl,$output->category_srl,"인터넷요금");
			$this->insertCategoryItemDefault($args,$output3->category_item_srl,$output->category_srl,"휴대폰사용료");
			$this->insertCategoryItemDefault($args,$output3->category_item_srl,$output->category_srl,"집전화요금");
			$this->insertCategoryItemDefault($args,$output3->category_item_srl,$output->category_srl,"인터넷전화요금");
			$this->insertCategoryItemDefault($args,$output3->category_item_srl,$output->category_srl,"기타");
			$output4 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"생활용품비");
			$this->insertCategoryItemDefault($args,$output4->category_item_srl,$output->category_srl,"가구");
			$this->insertCategoryItemDefault($args,$output4->category_item_srl,$output->category_srl,"가전");
			$this->insertCategoryItemDefault($args,$output4->category_item_srl,$output->category_srl,"주방");
			$this->insertCategoryItemDefault($args,$output4->category_item_srl,$output->category_srl,"욕실");
			$this->insertCategoryItemDefault($args,$output4->category_item_srl,$output->category_srl,"잡화");
			$this->insertCategoryItemDefault($args,$output4->category_item_srl,$output->category_srl,"기타");
			$output5 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"의복비");
			$this->insertCategoryItemDefault($args,$output5->category_item_srl,$output->category_srl,"의류비");
			$this->insertCategoryItemDefault($args,$output5->category_item_srl,$output->category_srl,"패션잡화");
			$this->insertCategoryItemDefault($args,$output5->category_item_srl,$output->category_srl,"세탁소비");
			$this->insertCategoryItemDefault($args,$output5->category_item_srl,$output->category_srl,"기타");
			$output6 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"미용비");
			$this->insertCategoryItemDefault($args,$output6->category_item_srl,$output->category_srl,"사우나");
			$this->insertCategoryItemDefault($args,$output6->category_item_srl,$output->category_srl,"화장품");
			$this->insertCategoryItemDefault($args,$output6->category_item_srl,$output->category_srl,"미용실비");
			$this->insertCategoryItemDefault($args,$output6->category_item_srl,$output->category_srl,"피부관리");
			$this->insertCategoryItemDefault($args,$output6->category_item_srl,$output->category_srl,"기타");
			$output7 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"건강비");
			$this->insertCategoryItemDefault($args,$output7->category_item_srl,$output->category_srl,"병원비");
			$this->insertCategoryItemDefault($args,$output7->category_item_srl,$output->category_srl,"약품비");
			$this->insertCategoryItemDefault($args,$output7->category_item_srl,$output->category_srl,"기타");
			$output8 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"문화비");
			$this->insertCategoryItemDefault($args,$output8->category_item_srl,$output->category_srl,"관람/입장료");
			$this->insertCategoryItemDefault($args,$output8->category_item_srl,$output->category_srl,"도서구입비");
			$this->insertCategoryItemDefault($args,$output8->category_item_srl,$output->category_srl,"기타");
			$output9 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"교육비");
			$this->insertCategoryItemDefault($args,$output9->category_item_srl,$output->category_srl,"등록금");
			$this->insertCategoryItemDefault($args,$output9->category_item_srl,$output->category_srl,"학원비");
			$this->insertCategoryItemDefault($args,$output9->category_item_srl,$output->category_srl,"교재비");
			$this->insertCategoryItemDefault($args,$output9->category_item_srl,$output->category_srl,"학용품비");
			$this->insertCategoryItemDefault($args,$output9->category_item_srl,$output->category_srl,"기타");
			$output10 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"육아비");
			$this->insertCategoryItemDefault($args,$output10->category_item_srl,$output->category_srl,"육아용품");
			$this->insertCategoryItemDefault($args,$output10->category_item_srl,$output->category_srl,"기타");
			$output11 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"경조사");
			$this->insertCategoryItemDefault($args,$output11->category_item_srl,$output->category_srl,"경조사비");
			$this->insertCategoryItemDefault($args,$output11->category_item_srl,$output->category_srl,"선물");
			$this->insertCategoryItemDefault($args,$output11->category_item_srl,$output->category_srl,"기타");
			$output12 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"모임");
			$this->insertCategoryItemDefault($args,$output12->category_item_srl,$output->category_srl,"회비");
			$this->insertCategoryItemDefault($args,$output12->category_item_srl,$output->category_srl,"유흥비");
			$this->insertCategoryItemDefault($args,$output12->category_item_srl,$output->category_srl,"기타");
			$output13 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"교통비");
			$this->insertCategoryItemDefault($args,$output13->category_item_srl,$output->category_srl,"지하철");
			$this->insertCategoryItemDefault($args,$output13->category_item_srl,$output->category_srl,"택시");
			$this->insertCategoryItemDefault($args,$output13->category_item_srl,$output->category_srl,"시내버스");
			$this->insertCategoryItemDefault($args,$output13->category_item_srl,$output->category_srl,"고속버스");
			$this->insertCategoryItemDefault($args,$output13->category_item_srl,$output->category_srl,"열차");
			$this->insertCategoryItemDefault($args,$output13->category_item_srl,$output->category_srl,"비행기");
			$this->insertCategoryItemDefault($args,$output13->category_item_srl,$output->category_srl,"기타");
			$output14 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"용돈");
			$this->insertCategoryItemDefault($args,$output14->category_item_srl,$output->category_srl,"부모");
			$this->insertCategoryItemDefault($args,$output14->category_item_srl,$output->category_srl,"자녀");
			$this->insertCategoryItemDefault($args,$output14->category_item_srl,$output->category_srl,"처가");
			$this->insertCategoryItemDefault($args,$output14->category_item_srl,$output->category_srl,"기타");
			$output15 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"세금");
			$this->insertCategoryItemDefault($args,$output15->category_item_srl,$output->category_srl,"근로소득세");
			$this->insertCategoryItemDefault($args,$output15->category_item_srl,$output->category_srl,"사업소득세");
			$this->insertCategoryItemDefault($args,$output15->category_item_srl,$output->category_srl,"양도소득세");
			$this->insertCategoryItemDefault($args,$output15->category_item_srl,$output->category_srl,"교육세");
			$this->insertCategoryItemDefault($args,$output15->category_item_srl,$output->category_srl,"농특세");
			$this->insertCategoryItemDefault($args,$output15->category_item_srl,$output->category_srl,"주민세");
			$this->insertCategoryItemDefault($args,$output15->category_item_srl,$output->category_srl,"재산세");
			$this->insertCategoryItemDefault($args,$output15->category_item_srl,$output->category_srl,"증여세");
			$this->insertCategoryItemDefault($args,$output15->category_item_srl,$output->category_srl,"취득세");
			$output16 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"카드대금");
			$this->insertCategoryItemDefault($args,$output16->category_item_srl,$output->category_srl,"카드대금");
			$output17 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"자동차");
			$this->insertCategoryItemDefault($args,$output17->category_item_srl,$output->category_srl,"자동차할부금");
			$this->insertCategoryItemDefault($args,$output17->category_item_srl,$output->category_srl,"자동차보험료");
			$this->insertCategoryItemDefault($args,$output17->category_item_srl,$output->category_srl,"자동차장식품");
			$this->insertCategoryItemDefault($args,$output17->category_item_srl,$output->category_srl,"자동차소모품");
			$this->insertCategoryItemDefault($args,$output17->category_item_srl,$output->category_srl,"세차비");
			$this->insertCategoryItemDefault($args,$output17->category_item_srl,$output->category_srl,"주유비");
			$this->insertCategoryItemDefault($args,$output17->category_item_srl,$output->category_srl,"통행료");
			$this->insertCategoryItemDefault($args,$output17->category_item_srl,$output->category_srl,"주차비");
			$this->insertCategoryItemDefault($args,$output17->category_item_srl,$output->category_srl,"자동차검사비");
			$this->insertCategoryItemDefault($args,$output17->category_item_srl,$output->category_srl,"자동차수리비");
			$this->insertCategoryItemDefault($args,$output17->category_item_srl,$output->category_srl,"자동차세금");
			$this->insertCategoryItemDefault($args,$output17->category_item_srl,$output->category_srl,"범칙금");
			$this->insertCategoryItemDefault($args,$output17->category_item_srl,$output->category_srl,"기타");
			$output18 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"저축");
			$this->insertCategoryItemDefault($args,$output18->category_item_srl,$output->category_srl,"예금");
			$this->insertCategoryItemDefault($args,$output18->category_item_srl,$output->category_srl,"적금");
			$this->insertCategoryItemDefault($args,$output18->category_item_srl,$output->category_srl,"펀드");
			$this->insertCategoryItemDefault($args,$output18->category_item_srl,$output->category_srl,"연금");
			$this->insertCategoryItemDefault($args,$output18->category_item_srl,$output->category_srl,"기타");
			$output19 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"보험");
			$this->insertCategoryItemDefault($args,$output19->category_item_srl,$output->category_srl,"종신보험");
			$this->insertCategoryItemDefault($args,$output19->category_item_srl,$output->category_srl,"상해보험");
			$this->insertCategoryItemDefault($args,$output19->category_item_srl,$output->category_srl,"의료보험");
			$this->insertCategoryItemDefault($args,$output19->category_item_srl,$output->category_srl,"연금보험");
			$this->insertCategoryItemDefault($args,$output19->category_item_srl,$output->category_srl,"기타");
			$output20 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"기부");
			$this->insertCategoryItemDefault($args,$output20->category_item_srl,$output->category_srl,"정치단체");
			$this->insertCategoryItemDefault($args,$output20->category_item_srl,$output->category_srl,"종교단체");
			$this->insertCategoryItemDefault($args,$output20->category_item_srl,$output->category_srl,"시민단체");
			$this->insertCategoryItemDefault($args,$output20->category_item_srl,$output->category_srl,"기타");
			$output21 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"기타");
			$this->insertCategoryItemDefault($args,$output21->category_item_srl,$output->category_srl,"기타");

            return $output;
        }
            
    	/**
         * 거래분류 기본값 입력
         * 
         * @param $args
         * @return unknown_type
         */
        function setDefaultDealCategoryData($args){
			$output = $this->insertCategoryDefault($args,"거래분류");
			$output1 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"기타");
        	return $output;
        }
                
        /**
         * 현금계정 기본값 입력
         * 
         * @param $args
         * @return unknown_type
         */
        function setDefaultCashAccountData($args){
			$output = $this->insertCategoryDefault($args,"현금계정분류");
			$output1 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"내지갑");
			$output1 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"비상금");
        	return $output;
        }
        
    	/**
         * 은행계정 기본값 입력
         * 
         * @param $args
         * @return unknown_type
         */
        function setDefaultBankData($args){
			$output = $this->insertCategoryDefault($args,"은행계정분류");
			$output1 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"보통예금");
			$output1 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"정기적금");
			$output1 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"정기예금");
			$output1 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"대출");
			$output1 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"기타예금");
        	return $output;
        }
        
    	/**
         * 자산계정 기본값 입력
         * 
         * @param $args
         * @return unknown_type
         */
        function setDefaultAssetData($args){
			$output = $this->insertCategoryDefault($args,"자산계정분류");
			$output1 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"부동산");
			$output1 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"자동차");
			$output1 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"회원권");
			$output1 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"귀금속");
			$output1 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"빌려준돈");
			$output1 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"기타자산");
        	return $output;
        }
        
    	/**
         * 카드계정분류 기본값 입력
         * 
         * @param $args
         * @return unknown_type
         */
        function setDefaultCardData($args){
			$output = $this->insertCategoryDefault($args,"카드계정분류");
			$output1 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"신용카드");
			$output1 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"카드론");
        	return $output;
        }
        
    	/**
         * 부채계정분류  기본값 입력
         * 
         * @param $args
         * @return unknown_type
         */
        function setDefaultDebtData($args){
			$output = $this->insertCategoryDefault($args,"부채계정분류");
			$output1 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"빌린돈");
			$output1 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"기타부채");
        	return $output;
        }
        
    	/**
         * 거래처분류 기본값 입력
         * 
         * @param $args
         * @return unknown_type
         */
        function setDefaultCustomerData($args){
			$output = $this->insertCategoryDefault($args,"거래처분류");
			$output1 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"대형마트");
			$output1 = $this->insertCategoryItemDefault($args,0,$output->category_srl,"기타");
        	return $output;
        }
    	
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
    	/**
         * @brief 권한 가져오기
         **/
        function getModuleGrants($args) {
			// 쿼리실행
            $output = executeQueryArray('cashbook.getModuleGrants', $args);
            return $output;
        }
        
    	/**
         * @brief 회원이 속한 그룹 가져오기
         **/
        function getMemberGroupList($args) {
            $output = executeQueryArray('cashbook.getMemberGroupList', $args);
            return $output;
        }
        
        
        
        /**
         * 자산합구하기
         * @param $args
         * @return unknown_type
         */
        function getAssetSum($args){
        	// 계정목록가져오기
            $output = $this->getAccountListAll($args);

            // 자산계산
            $asset_total = 0;     // 자산의 합
            foreach($output->data as $key => $val) {
            	$amount_total = 0;    // 거래금액의 합을 자산이 시작할 때 초기화한다.	            	
            	$args->account_srl = $val->account_srl;
				// 계정에 해당하는 거래 가져오기
				$outputDeal = $this->getDealByAccount($args);
				if(count($outputDeal->data)>1) {
	            	foreach($outputDeal->data as $k => $v) {
						$amount_total += $v->amount;
					}
				} else {
					$amount_total += $outputDeal->data->amount;
				}
				$val->amount_total = $amount_total;			
				$account_list[$key] = $val;
                    
                // 계정기본금액을 자산에 더하기
                $asset_total += $val->amount;                 
                // 거래금액을 자산에 더하기
                $asset_total += $amount_total;
                
                $assetSum->account_list = $account_list;             
                $assetSum->asset_total = $asset_total;
            }
            
            return $assetSum;
        }
        
        /**
         * 오늘로부터 7일간거래 가져오기
         * @return unknown_type
         */
        function getLastWeekDeal(){
        	// 오늘로부터 7일전 날짜 가져오기
            $timestamp=mktime(0,0,0,date("n"),date("j"),date("Y"));
			$last_week=strtotime("-1 week",$timestamp);
			$lastWeekYear = date("Y",$last_week);
			$lastWeekMonth = date("n",$last_week);
			$lastWeekDay = date("j",$last_week); 

            $tempYear = date("Y");
            $tempMonth = date("n");
            if($tempMonth < 10){
            	$tempMonth = "0".$tempMonth;
            }
            $tempDay = date("j");
        	if($tempMonth < 10){
            	$tempDay = "0".$tempDay;
            }
            $args->sort_index = "input_date";	// 정렬 컬럼
            $args->start_day = $lastWeekYear.$lastWeekMonth.$lastWeekDay;
            $args->end_day = $tempYear.$tempMonth.$tempDay;
            $output = $this->getDealListByPeriod($args);
            $deal_week_list;
            for($i=0;$i<7;$i++){
            	$val = null;	// 초기화
            	$tmpTimestamp = mktime(0,0,0,date("n"),date("j")-$i,date("Y"));
				$tmpTime=strtotime("-1 day",$tmpTimestamp);
				$tmpYear = date("Y",$tmpTime);
				$tmpMonth = date("n",$tmpTime);
				$tmpDay = date("j",$tmpTime); 
	            if($tmpMonth < 10){
	            	$tmpMonth = "0".$tmpMonth;
	            }
	        	if($tmpDay < 10){
	            	$tmpDay = "0".$tmpDay;
	            }
            	$val->day = $tmpYear.$tmpMonth.$tmpDay;
            	$incomeCount = 0;
            	$outgoCount = 0;
            	$transferCount = 0;
            	// 수입/지출/이체 수 계산하기
            	if($output->data!=null){
	            	foreach($output->data as $n => $v){
            			if($v->day==$val->day && $v->amount>0){
            				$incomeCount++;
            			}
	            		if($v->day==$val->day && $v->amount<0){
            				$incomeCount++;
            			}
	            		if($v->day==$val->day && $v->transfer_srl>0){
            				$transferCount++;
            			}
	            	}
            	}
            	$val->income_count = $incomeCount;
            	$val->outgo_count = $outgoCount;
            	$val->transfer_count = $transferCount/2;
                $deal_week_list[$i] = $val;
                $deal_list->data = $deal_week_list;
            }
            return $deal_list;
        }

    	/**
         * 검색조건가져오기
         * @param $args
         * @return unknown_type
         */
        function getCustomerSearchParameter($search_target,$search_keyword){
        	if($search_target && $search_keyword) {
        		switch($search_target) {
                    case 'category' :
                            if($search_keyword) $search_keyword = str_replace(' ','%',$search_keyword);
                            $args->s_category_name = $search_keyword;
                        break;
                    case 'title' :
                            if($search_keyword) $search_keyword = str_replace(' ','%',$search_keyword);
                            $args->s_title = $search_keyword;
                        break;
                    case 'phone' :
                            if($search_keyword) $search_keyword = str_replace(' ','%',$search_keyword);
                            $args->s_phone = $search_keyword;
                        break;
                    case 'memo' :
                            if($search_keyword) $search_keyword = str_replace(' ','%',$search_keyword);
                            $args->s_memo = $search_keyword;
                        break;
                    case 'input_date' :
                            $args->s_input_date = ereg_replace("[^0-9]","",$search_keyword);
                        break;
                    case 'input_date_more' :
                            $args->s_input_date_more = substr(ereg_replace("[^0-9]","",$search_keyword) . '00000000000000',0,14);
                        break;
                    case 'input_date_less' :
                            $args->s_input_date_less = substr(ereg_replace("[^0-9]","",$search_keyword) . '00000000000000',0,14);
                        break;
                }
            }
            
            return $args;
        }
        
        /**
         * 거래검색조건가져오기
         * @param $args
         * @return unknown_type
         */
        function getDealSearchParameter($search_target,$search_keyword){
        	if($search_target && $search_keyword) {
                switch($search_target) {
                    case 'day' :
                            $args->s_day = ereg_replace("[^0-9]","",$search_keyword);
                        break;
                    case 'title' :
                            if($search_keyword) $search_keyword = str_replace(' ','%',$search_keyword);
                            $args->s_title = $search_keyword;
                        break;
                    case 'amount' :
                            $args->s_amount = ereg_replace("[^0-9]","",$search_keyword);
                        break;
                    case 'amount_more' :
                            $args->s_amount_more = ereg_replace("[^0-9]","",$search_keyword);
                        break;
                    case 'amount_less' :
                            $args->s_amount_less = ereg_replace("[^0-9]","",$search_keyword);
                        break;
                    case 'item' :
                            if($search_keyword) $search_keyword = str_replace(' ','%',$search_keyword);
                            $args->s_item_name = $search_keyword;
                        break;
                    case 'category' :
                            if($search_keyword) $search_keyword = str_replace(' ','%',$search_keyword);
                            $args->s_category_name = $search_keyword;
                        break;
                    case 'customer' :
                            if($search_keyword) $search_keyword = str_replace(' ','%',$search_keyword);
                            $args->s_customer_name = $search_keyword;
                        break;
                    case 'tag' :
                            if($search_keyword) $search_keyword = str_replace(' ','%',$search_keyword);
                            $args->s_tag = $search_keyword;
                        break;
                    case 'input_date' :
                            $args->s_input_date = ereg_replace("[^0-9]","",$search_keyword);
                        break;
                    case 'input_date_more' :
                            $args->s_input_date_more = substr(ereg_replace("[^0-9]","",$search_keyword) . '00000000000000',0,14);
                        break;
                    case 'input_date_less' :
                            $args->s_input_date_less = substr(ereg_replace("[^0-9]","",$search_keyword) . '00000000000000',0,14);
                        break;
                }
            }
            
            return $args;
        }

        
    	
        
        /**
         * 파라미터로 넘어온 값이 없을 경우 현재 년도를 반환한다.
         * 
         * @param $args
         * @return unknown_type
         */
        function getYear($args){
        	$todayYear = date("Y");	// 당일년도
        	if($args->pYear==null){
        		$pYear = $todayYear;
        	} else {
        		$pYear= $args->pYear;
        	}
        	return $pYear;
        }
        
        
    	/**
         * 파라미터로 넘어온 값이 없을 경우 현재 월을 반환한다.
         * 
         * @param $args
         * @return unknown_type
         */
        function getMonth($args){
        $todayMonth = date("n");	// 당일월
        	if($args->pMonth==null){
        		$pMonth = $todayMonth;
        	} else {
        		$pMonth= $args->pMonth;
        	}
        	if($pMonth<10){
				$tempMonth = "0".$pMonth;
			} else {
				$tempMonth = $pMonth;
			}
			return $tempMonth;
        }
        
    	/**
         * 파라미터로 넘어온 값이 없을 경우 현재 일을 반환한다.
         * 
         * @param $args
         * @return unknown_type
         */
        function getDay($args){
        	$todayDay = date("j");	// 당일일
        	if($args->pDay==null){
        		$pDay = $todayDay;
        	} else {
        		$pDay= $args->pDay;
        	}
        	if($pDay<10){
				$tempDay = "0".$pDay;
			} else {
				$tempDay = $pDay;
			}
        	return $tempDay;
        }

    	
        
        /**
         * 금지 아이디인지 확인
         * 
         * @param $args
         * @return unknown_type
         */
        function isDeniedID($user_id) {
            $args->user_id = $user_id;
            $output = executeQuery('module.getDeniedID', $args);
            if($output->data->count) return true;
            return false;
        }
        
        
    }
?>