<?php

    /**
     * @class statisticsModel
     * @author mooo (hhyoon@gmail.com)
     * @brief statistics 모듈의 Model class
     **/

    class statisticsModel extends statistics {

        var $agents = array();
        var $countries = array();

        function init() {
        }


        /**
         * @brief get list of agents
         */
        function getAgents() {

            if (count($this->agents) > 0) return true;

            // read cache file
            $cache_file = "./files/cache/statistics/agents.php";

            // if no cache file, make it
            if (!file_exists($cache_file)) {
                $oStatisticsController = &getController('statistics');
                if (!$oStatisticsController->makeCacheAgents()) return false;
            }

            @include_once($cache_file);

            if (!$array_agents or !count($array_agents)) return false;

            /*
            // using ordinary associative array
            foreach ($array_agents as $val) {
                unset($agent);

                $agent->browser = $val['browser'];
                $agent->user_agent = $val['user_agent'];
                $agent->agent_type = $val['agent_type'];

                $this->agents[] = $agent;
            }
            */

            // using object cast
            $this->agents = $array_agents;

        }

        /**
         * @brief get list of countries
         */
        function getCountries() {

            if (count($this->countries) > 0) return true;

            // read cache file
            $cache_file = "./files/cache/statistics/countries.php";

            // if no cache file, make it
            if (!file_exists($cache_file)) {
                $oStatisticsController = &getController('statistics');
                if (!$oStatisticsController->makeCacheCountries()) return false;
            }

            @include_once($cache_file);

            if (!$array_countries or !count($array_countries)) return false;

            // using object cast
            $this->countries = $array_countries;

        }

        /**
         * @brief check this is the first access for today
         **/
        function isUniqueVisitor($site_srl = 0) {

            $args->regdate = date('Ymd');
            $args->site_srl = $site_srl;
            $args->ipaddress = $_SERVER['REMOTE_ADDR'];
            $output = executeQuery('statistics.isNewVisitor', $args);
            return $output->data->count ? true : false;
        }

        /**
         * @brief check this is the first access up to now
         **/
        function isNewVisitor($site_srl = 0) {

            $args->regdate = 0;
            $args->site_srl = $site_srl;
            $args->ipaddress = $_SERVER['REMOTE_ADDR'];
            $output = executeQuery('statistics.isNewVisitor', $args);
            return $output->data->count ? true : false;
        }

        /**
         * @brief check this is the first block for today
         **/
        function isUniqueBlock($site_srl = 0) {

            $args->regdate = date('Ymd');
            $args->site_srl = $site_srl;
            $args->ipaddress = $_SERVER['REMOTE_ADDR'];
            $output = executeQuery('statistics.isNewBlock', $args);
            return $output->data->count ? true : false;
        }


        /**
         * @brief check for enable blocking
         **/
        function sendHttpHeader($code = '404', $go_die = true) {
            $header = array(
                '403' => 'Forbidden',
                '404' => 'Not Found',
            );

            $header_message = sprintf("%s %s", $code, $header[$code]);

            header("HTTP/1.0 ". $header_message);
            header("Status: ". $header_message);

            if ($go_die) {
                header('Content-Type: text/html; charset=iso-8859-1');
                $html = sprintf(
                    '<!DOCTYPE HTML PUBLIC "-//IETF//DTD HTML 2.0//EN"><html lang="en" xml:lang="en"><head><title>'.
                    '%s</title></head><body><h1>%s</h1><p>Something is wrong with your request!</p></body></html>',
                    $header_message, $header[$code]
                );
                die($html);
            }
        }

        /**
         * @brief check for enable blocking
         **/
        function checkValidAccess() {
            if ($_SERVER['REMOTE_ADDR'] == '::1') return false;
            if (!$_SERVER['HTTP_USER_AGENT']) return false;

            return true;
        }

        /**
         * @brief check for enable blocking
         **/
        function checkEnableBlocking() {

            if ($_SERVER['REMOTE_ADDR'] == '::1') return false;
            if (!$_SERVER['HTTP_USER_AGENT']) return false;

            $oModuleModel = &getModel('module');
            $config = $oModuleModel->getModuleConfig('statistics');
            if (!$config) return false;

            // filter for user agent
            if ($config->block_agent == 'Y' and $config->filter_agents) {
                $filters = explode(',', $config->filter_agents);
                foreach ($filters as $filter) {
                    if (strpos($_SERVER['HTTP_USER_AGENT'], $filter) !== false)
                        $this->sendHttpHeader();
                }
            }

            // filter for ip address
            if ($config->block_ipaddress == 'Y' and $config->filter_ipaddresses) {
                $filters = explode(',', $config->filter_ipaddresses);
                foreach ($filters as $filter) {
                    if ($this->ipfilter($_SERVER['REMOTE_ADDR'], $filter))
                        $this->sendHttpHeader();
                }
            }

            // check configuration
            if ($config->use_block != 'Y') return false;

            // check access count during block_interval value

            return true;
        }

        /**
         * @brief check for enable logging
         **/
        function checkEnableLogging() {

            if ($_SERVER['REMOTE_ADDR'] == '::1') return false;
            if (!$_SERVER['HTTP_USER_AGENT']) return false;
            if (!in_array($_SERVER['REQUEST_METHOD'], array('GET', 'POST'))) return false;

            $oModuleModel = &getModel('module');
            $config = $oModuleModel->getModuleConfig('statistics');
            if (!$config) return false;

            // check admin
            if ($config->allow_log_admin != 'Y') {
                $logged_info = Context::get('logged_info');
                if ($logged_info and $logged_info->is_admin == 'Y') return false;
            }

            // filter for user agent
            if ($config->block_agent == 'N' and $config->filter_agent == 'Y' and $config->filter_agents) {
                $filters = explode(',', $config->filter_agents);
                foreach ($filters as $filter) {
                    if (strpos($_SERVER['HTTP_USER_AGENT'], $filter) !== false) return false;
                }
            }

            // filter for ip address
            if ($config->block_ipaddress == 'N' and $config->filter_ipaddress == 'Y' and $config->filter_ipaddresses) {
                $filters = explode(',', $config->filter_ipaddresses);
                foreach ($filters as $filter) {
                    if ($this->ipfilter($_SERVER['REMOTE_ADDR'], $filter)) return false;
                }
            }

            return true;
        }

        /**
         * @brief get a keyword from referer array
         **/
        function getRefererKeyword($referer) {

            if (!$referer['host'] or !$referer['query']) return '';

            $keyword = '';
            $queries = explode('&', $referer['query']);
            foreach ($queries as $query) {
                if (preg_match('/^(query|q|p)=(.+)$/i', $query, $matches)) {
                    //$keyword = Context::convertEncodingStr(removeHackTag($matches[2]));
                    $keyword = $this->_urldecode($matches[2]);
                    break;
                }
            }

            return trim($keyword);
        }

        /**
         * @brief ip filter
         *
         * refer from http://kr2.php.net/ip2long
         **/
        function ipfilter($ipaddress, $filter) {

            // netmask
            if (strpos($filter, '/') !== false) {
                $ip_addr = decbin($this->_ip2long($ipaddress));

                $network = explode("/", $filter);
                $net_addr = decbin($this->_ip2long($network[0]));
                $cidr = $network[1];

                if (substr($net_addr, 0, $cidr) == substr($ip_addr, 0, $cidr))
                    return true;
            }
            // ip address
            else if ($ipaddress == $filter)
                return true;

            return false;
        }

        /**
         * @brief modification of ip2long() function
         *
         * refer from http://kr2.php.net/ip2long
         **/
        function _ip2long($address) {

            if (is_numeric($address)) {
                return sprintf("%u", floatval($address));
            } else {
                return sprintf("%u", floatval(ip2long($address)));
            }
        }

        /**
         * @brief check agent type and name
         **/
        function getAgentValue($user_agent) {
            if (!$this->agents) $this->getAgents();
            if (!$this->agents) return null;

            foreach ($this->agents as $agent) {
                if (strpos($user_agent, $agent->user_agent) !== false) return $agent;
            }

            return null;
        }

        /**
         * @brief modification of urldecode() function
         **/
        function _urldecode($str) {

            $str = Context::convertEncodingStr(urldecode($str));
            $str = preg_replace("/%u([0-9a-f]{3,4})/i", "&#x\\1;", $str); ///< for ie6
            $str = html_entity_decode($str, null, 'UTF-8'); ///< for ie6

            return trim($str);
        }

        /**
         * @brief get a icon image tag for robots and tools
         **/
        function getBrowserIcon($agent_type) {
            $icon = '';

            if ($agent_type == 'R')
                $icon .= ' <img src="modules/statistics/tpl/images/bug.gif" class="is_robot" title="Robot" alt="Robot" width="10" height="10" />';
            else if ($agent_type == 'T')
                $icon .= ' <img src="modules/statistics/tpl/images/basket.gif" class="is_robot" title="Tool" alt="Tool" width="10" height="10" />';

            return $icon;
        }

        /**
         * @brief get a country name from ip address
         **/
        function getCountryName($ipaddress) {

            // refer from http://www.maxmind.com/app/php
            @include_once('./modules/statistics/libs/geoip/geoip.inc');

            $country = 'Unknown';
            $gi = geoip_open('./modules/statistics/libs/geoip/data/GeoIP.dat', GEOIP_STANDARD);
            if ($gi) {
                $country = geoip_country_name_by_addr($gi, $ipaddress);
                geoip_close($gi);
            }

            return $country;
        }

        /**
         * @brief get a country name from ip address
         **/
        function getCountryCode($ipaddress) {

            // refer from http://www.maxmind.com/app/php
            @include_once('./modules/statistics/libs/geoip/geoip.inc');

            $code = '';
            $gi = geoip_open('./modules/statistics/libs/geoip/data/GeoIP.dat', GEOIP_STANDARD);
            if ($gi) {
                $code = geoip_country_code_by_addr($gi, $ipaddress);
                geoip_close($gi);
            }

            return $code;
        }

        /**
         * @brief check agent type and name
         **/
        function getCountryNameFromCode($code) {
            if (!$this->countries) $this->getCountries();
            if (!$this->countries) return '';

            foreach ($this->countries as $country) {
                if ($code == $country->code) return $country->name;
            }

            return 'Unknown';
        }


        /**
         * @brief list of statistics
         **/
        function getListStatisticsTotal($obj) {

            $args->site_srl = $obj->site_srl ? $obj->site_srl : 0;
            $args->regdate = 0;

            $output = executeQuery('statistics.getListStatisticsTotal', $args);

            // 결과가 없거나 오류 발생시 그냥 return
            if (!$output->toBool() || !count($output->data)) return new Object();

            return $output;
        }

        /**
         * @brief list of statistics day
         **/
        function getListStatisticsDay($obj) {

            $args->site_srl = $obj->site_srl ? $obj->site_srl : 0;

            $args->sort_index = $obj->sort_index ? $obj->sort_index : 'regdate';
            $args->order_type = $obj->order_type ? $obj->order_type : 'desc';
            $args->page = $obj->page ? $obj->page : 1;
            $args->list_count = $obj->list_count ? $obj->list_count : 30;
            $args->page_count = $obj->page_count ? $obj->page_count : 10;

            $output = executeQuery('statistics.getListStatisticsDay', $args);

            // 결과가 없거나 오류 발생시 그냥 return
            if (!$output->toBool() || !count($output->data)) return new Object();

            return $output;
        }

        /**
         * @brief list of statistics month
         **/
        function getListStatisticsMonth($obj) {

            $args->site_srl = $obj->site_srl ? $obj->site_srl : 0;

            $args->sort_index = $obj->sort_index ? $obj->sort_index : 'regdate';
            $args->order_type = $obj->order_type ? $obj->order_type : 'desc';
            $args->page = $obj->page ? $obj->page : 1;
            $args->list_count = $obj->list_count ? $obj->list_count : 30;
            $args->page_count = $obj->page_count ? $obj->page_count : 10;

            $output = executeQuery('statistics.getListStatisticsMonth', $args);

            // 결과가 없거나 오류 발생시 그냥 return
            if (!$output->toBool() || !count($output->data)) return new Object();

            return $output;
        }

        /**
         * @brief list of statistics year
         **/
        function getListStatisticsYear($obj) {

            $args->site_srl = $obj->site_srl ? $obj->site_srl : 0;

            $args->sort_index = $obj->sort_index ? $obj->sort_index : 'regdate';
            $args->order_type = $obj->order_type ? $obj->order_type : 'desc';
            $args->page = $obj->page ? $obj->page : 1;
            $args->list_count = $obj->list_count ? $obj->list_count : 30;
            $args->page_count = $obj->page_count ? $obj->page_count : 10;

            $output = executeQuery('statistics.getListStatisticsYear', $args);

            // 결과가 없거나 오류 발생시 그냥 return
            if (!$output->toBool() || !count($output->data)) return new Object();

            return $output;
        }


        /**
         * @brief list of statistics log
         **/
        function getLogList($obj) {

            $args->site_srl = $obj->site_srl ? $obj->site_srl : 0;
            if ($obj->agent_type != 'A')
                $args->agent_type = $obj->agent_type ? $obj->agent_type : 'U';

            $args->sort_index = $obj->sort_index ? $obj->sort_index : 'regdate';
            $args->page = $obj->page ? $obj->page : 1;
            $args->list_count = $obj->list_count ? $obj->list_count : 30;
            $args->page_count = $obj->page_count ? $obj->page_count : 10;

            $output = executeQuery('statistics.getListLog', $args);

            // 결과가 없거나 오류 발생시 그냥 return
            if (!$output->toBool() || !count($output->data)) return new Object();

            return $output;
        }


        /**
         * @brief get total count for agents
         **/
        function getTotalCountAgents($obj) {

            $args->site_srl = $obj->site_srl ? $obj->site_srl : 0;
            if ($obj->agent_type != 'A')
                $args->agent_type = $obj->agent_type ? $obj->agent_type : 'U';

            if (isset($obj->from_date) and isset($obj->to_date)) {
                $args->from_date = $obj->from_date;
                $args->to_date = $obj->to_date;
            }
            else
                $args->regdate = isset($obj->regdate) ? $obj->regdate : 0;

            $output = executeQuery('statistics.getTotalCountAgents', $args);

            // 결과가 없거나 오류 발생시 그냥 return
            if (!$output->toBool() || !count($output->data)) return new Object();

            return $output;
        }

        /**
         * @brief list of agents
         **/
        function getListAgents($obj) {

            $args->site_srl = $obj->site_srl ? $obj->site_srl : 0;
            if ($obj->agent_type != 'A')
                $args->agent_type = $obj->agent_type ? $obj->agent_type : 'U';

            if (isset($obj->from_date) and isset($obj->to_date)) {
                $args->from_date = $obj->from_date;
                $args->to_date = $obj->to_date;
            }
            else
                $args->regdate = isset($obj->regdate) ? $obj->regdate : 0;

            $args->sort_index = $obj->sort_index ? $obj->sort_index : 'visitors';
            $args->sort_index2 = $obj->sort_index2 ? $obj->sort_index2 : 'pageviews';
            $args->page = $obj->page ? $obj->page : 1;
            $args->list_count = $obj->list_count ? $obj->list_count : 30;
            $args->page_count = $obj->page_count ? $obj->page_count : 10;

            $output = executeQuery('statistics.getListAgents', $args);

            // 결과가 없거나 오류 발생시 그냥 return
            if (!$output->toBool() || !count($output->data)) return new Object();

            return $output;
        }

        /**
         * @brief list of browsers
         **/
        function getListBrowsers($obj) {

            $args->site_srl = $obj->site_srl ? $obj->site_srl : 0;
            if ($obj->agent_type != 'A')
                $args->agent_type = $obj->agent_type ? $obj->agent_type : 'U';

            if (isset($obj->from_date) and isset($obj->to_date)) {
                $args->from_date = $obj->from_date;
                $args->to_date = $obj->to_date;
            }
            else
                $args->regdate = isset($obj->regdate) ? $obj->regdate : 0;

            $args->sort_index = $obj->sort_index ? $obj->sort_index : 'visitors';
            $args->sort_index2 = $obj->sort_index2 ? $obj->sort_index2 : 'pageviews';
            $args->page = $obj->page ? $obj->page : 1;
            $args->list_count = $obj->list_count ? $obj->list_count : 30;
            $args->page_count = $obj->page_count ? $obj->page_count : 10;

            $output = executeQuery('statistics.getListBrowsers', $args);

            // 결과가 없거나 오류 발생시 그냥 return
            if (!$output->toBool() || !count($output->data)) return new Object();

            return $output;
        }

        /**
         * @brief get total count for visitors
         **/
        function getTotalCountVisitors($obj) {

            $args->site_srl = $obj->site_srl ? $obj->site_srl : 0;
            if ($obj->agent_type != 'A')
                $args->agent_type = $obj->agent_type ? $obj->agent_type : 'U';

            if (isset($obj->from_date) and isset($obj->to_date)) {
                $args->from_date = $obj->from_date;
                $args->to_date = $obj->to_date;
            }
            else
                $args->regdate = isset($obj->regdate) ? $obj->regdate : 0;

            $output = executeQuery('statistics.getTotalCountVisitors', $args);

            // 결과가 없거나 오류 발생시 그냥 return
            if (!$output->toBool() || !count($output->data)) return new Object();

            return $output;
        }

        /**
         * @brief get total count for mids
         **/
        function getTotalCountMidPageviews($obj) {

            $args->site_srl = $obj->site_srl ? $obj->site_srl : 0;

            if (isset($obj->from_date) and isset($obj->to_date)) {
                $args->from_date = $obj->from_date;
                $args->to_date = $obj->to_date;
            }
            else
                $args->regdate = isset($obj->regdate) ? $obj->regdate : 0;

            $output = executeQuery('statistics.getTotalCountMidPageviews', $args);

            // 결과가 없거나 오류 발생시 그냥 return
            if (!$output->toBool() || !count($output->data)) return new Object();

            return $output;
        }

        /**
         * @brief list of mids
         **/
        function getListMidPageviews($obj) {

            $args->site_srl = $obj->site_srl ? $obj->site_srl : 0;

            if (isset($obj->from_date) and isset($obj->to_date)) {
                $args->from_date = $obj->from_date;
                $args->to_date = $obj->to_date;
            }
            else
                $args->regdate = isset($obj->regdate) ? $obj->regdate : 0;

            $args->sort_index = $obj->sort_index ? $obj->sort_index : 'count';
            $args->page = $obj->page ? $obj->page : 1;
            $args->list_count = $obj->list_count ? $obj->list_count : 30;
            $args->page_count = $obj->page_count ? $obj->page_count : 10;

            $output = executeQuery('statistics.getListMidPageviews', $args);

            // 결과가 없거나 오류 발생시 그냥 return
            if (!$output->toBool() || !count($output->data)) return new Object();

            return $output;
        }

        /**
         * @brief get total count for visitors
         **/
        function getTotalCountCountryVisitors($obj) {

            $args->site_srl = $obj->site_srl ? $obj->site_srl : 0;
            if ($obj->agent_type != 'A')
                $args->agent_type = $obj->agent_type ? $obj->agent_type : 'U';

            if (isset($obj->from_date) and isset($obj->to_date)) {
                $args->from_date = $obj->from_date;
                $args->to_date = $obj->to_date;
            }
            else
                $args->regdate = isset($obj->regdate) ? $obj->regdate : 0;

            $output = executeQuery('statistics.getTotalCountCountryVisitors', $args);

            // 결과가 없거나 오류 발생시 그냥 return
            if (!$output->toBool() || !count($output->data)) return new Object();

            return $output;
        }

        /**
         * @brief list of visitors
         **/
        function getListCountryVisitors($obj) {

            $args->site_srl = $obj->site_srl ? $obj->site_srl : 0;
            if ($obj->agent_type != 'A')
                $args->agent_type = $obj->agent_type ? $obj->agent_type : 'U';

            if (isset($obj->from_date) and isset($obj->to_date)) {
                $args->from_date = $obj->from_date;
                $args->to_date = $obj->to_date;
            }
            else
                $args->regdate = isset($obj->regdate) ? $obj->regdate : 0;

            $args->sort_index = $obj->sort_index ? $obj->sort_index : 'count';
            $args->page = $obj->page ? $obj->page : 1;
            $args->list_count = $obj->list_count ? $obj->list_count : 30;
            $args->page_count = $obj->page_count ? $obj->page_count : 10;

            $output = executeQuery('statistics.getListCountryVisitors', $args);

            // 결과가 없거나 오류 발생시 그냥 return
            if (!$output->toBool() || !count($output->data)) return new Object();

            return $output;
        }

        /**
         * @brief list of visitors
         **/
        function getListVisitors($obj) {

            $args->site_srl = $obj->site_srl ? $obj->site_srl : 0;
            if ($obj->agent_type != 'A')
                $args->agent_type = $obj->agent_type ? $obj->agent_type : 'U';

            if (isset($obj->from_date) and isset($obj->to_date)) {
                $args->from_date = $obj->from_date;
                $args->to_date = $obj->to_date;
            }
            else
                $args->regdate = isset($obj->regdate) ? $obj->regdate : 0;

            $args->sort_index = $obj->sort_index ? $obj->sort_index : 'count';
            $args->page = $obj->page ? $obj->page : 1;
            $args->list_count = $obj->list_count ? $obj->list_count : 30;
            $args->page_count = $obj->page_count ? $obj->page_count : 10;

            $output = executeQuery('statistics.getListVisitors', $args);

            // 결과가 없거나 오류 발생시 그냥 return
            if (!$output->toBool() || !count($output->data)) return new Object();

            return $output;
        }

        /**
         * @brief get total count for blocks
         **/
        function getTotalCountBlocks($obj) {

            $args->site_srl = $obj->site_srl ? $obj->site_srl : 0;
            //if ($obj->agent_type != 'A')
            //    $args->agent_type = $obj->agent_type ? $obj->agent_type : 'U';

            if (isset($obj->from_date) and isset($obj->to_date)) {
                $args->from_date = $obj->from_date;
                $args->to_date = $obj->to_date;
            }
            else
                $args->regdate = isset($obj->regdate) ? $obj->regdate : 0;

            $output = executeQuery('statistics.getTotalCountBlocks', $args);

            // 결과가 없거나 오류 발생시 그냥 return
            if (!$output->toBool() || !count($output->data)) return new Object();

            return $output;
        }

        /**
         * @brief list of blocks
         **/
        function getListBlocks($obj) {

            $args->site_srl = $obj->site_srl ? $obj->site_srl : 0;
            //if ($obj->agent_type != 'A')
            //    $args->agent_type = $obj->agent_type ? $obj->agent_type : 'U';

            if (isset($obj->from_date) and isset($obj->to_date)) {
                $args->from_date = $obj->from_date;
                $args->to_date = $obj->to_date;
            }
            else
                $args->regdate = isset($obj->regdate) ? $obj->regdate : 0;

            $args->sort_index = $obj->sort_index ? $obj->sort_index : 'count';
            $args->page = $obj->page ? $obj->page : 1;
            $args->list_count = $obj->list_count ? $obj->list_count : 30;
            $args->page_count = $obj->page_count ? $obj->page_count : 10;

            $output = executeQuery('statistics.getListBlocks', $args);

            // 결과가 없거나 오류 발생시 그냥 return
            if (!$output->toBool() || !count($output->data)) return new Object();

            return $output;
        }

        /**
         * @brief get total count for referers
         **/
        function getTotalCountReferers($obj) {

            $args->site_srl = $obj->site_srl ? $obj->site_srl : 0;

            if (isset($obj->from_date) and isset($obj->to_date)) {
                $args->from_date = $obj->from_date;
                $args->to_date = $obj->to_date;
            }
            else
                $args->regdate = isset($obj->regdate) ? $obj->regdate : 0;

            $output = executeQuery('statistics.getTotalCountReferers', $args);

            // 결과가 없거나 오류 발생시 그냥 return
            if (!$output->toBool() || !count($output->data)) return new Object();

            return $output;
        }

        /**
         * @brief list of referers
         **/
        function getListReferers($obj) {

            $args->site_srl = $obj->site_srl ? $obj->site_srl : 0;

            if (isset($obj->from_date) and isset($obj->to_date)) {
                $args->from_date = $obj->from_date;
                $args->to_date = $obj->to_date;
            }
            else
                $args->regdate = isset($obj->regdate) ? $obj->regdate : 0;

            $args->sort_index = $obj->sort_index ? $obj->sort_index : 'count';
            $args->page = $obj->page ? $obj->page : 1;
            $args->list_count = $obj->list_count ? $obj->list_count : 30;
            $args->page_count = $obj->page_count ? $obj->page_count : 10;

            $output = executeQuery('statistics.getListReferers', $args);

            // 결과가 없거나 오류 발생시 그냥 return
            if (!$output->toBool() || !count($output->data)) return new Object();

            return $output;
        }

        /**
         * @brief get total count for keywords
         **/
        function getTotalCountKeywords($obj) {

            $args->site_srl = $obj->site_srl ? $obj->site_srl : 0;
            if ($obj->type) $args->type = $obj->type;

            if (isset($obj->from_date) and isset($obj->to_date)) {
                $args->from_date = $obj->from_date;
                $args->to_date = $obj->to_date;
            }
            else
                $args->regdate = isset($obj->regdate) ? $obj->regdate : 0;

            $output = executeQuery('statistics.getTotalCountKeywords', $args);

            // 결과가 없거나 오류 발생시 그냥 return
            if (!$output->toBool() || !count($output->data)) return new Object();

            return $output;
        }

        /**
         * @brief list of keywords
         **/
        function getListKeywords($obj) {

            $args->site_srl = $obj->site_srl ? $obj->site_srl : 0;
            if ($obj->type) $args->type = $obj->type;

            if (isset($obj->from_date) and isset($obj->to_date)) {
                $args->from_date = $obj->from_date;
                $args->to_date = $obj->to_date;
            }
            else
                $args->regdate = isset($obj->regdate) ? $obj->regdate : 0;

            $args->sort_index = $obj->sort_index ? $obj->sort_index : 'count';
            $args->page = $obj->page ? $obj->page : 1;
            $args->list_count = $obj->list_count ? $obj->list_count : 30;
            $args->page_count = $obj->page_count ? $obj->page_count : 10;

            $output = executeQuery('statistics.getListKeywords', $args);

            // 결과가 없거나 오류 발생시 그냥 return
            if (!$output->toBool() || !count($output->data)) return new Object();

            return $output;
        }



        /**
         * @brief get total count for pages
         **/
        function getTotalCountPages($obj) {

            $args->site_srl = $obj->site_srl ? $obj->site_srl : 0;

            if (isset($obj->from_date) and isset($obj->to_date)) {
                $args->from_date = $obj->from_date;
                $args->to_date = $obj->to_date;
            }
            else
                $args->regdate = isset($obj->regdate) ? $obj->regdate : 0;

            $output = executeQuery('statistics.getTotalCountPages', $args);

            // 결과가 없거나 오류 발생시 그냥 return
            if (!$output->toBool() || !count($output->data)) return new Object();

            return $output;
        }

        /**
         * @brief list of pages
         **/
        function getListPages($obj) {

            $args->site_srl = $obj->site_srl ? $obj->site_srl : 0;

            if (isset($obj->from_date) and isset($obj->to_date)) {
                $args->from_date = $obj->from_date;
                $args->to_date = $obj->to_date;
            }
            else
                $args->regdate = isset($obj->regdate) ? $obj->regdate : 0;

            $args->sort_index = $obj->sort_index ? $obj->sort_index : 'count';
            $args->page = $obj->page ? $obj->page : 1;
            $args->list_count = $obj->list_count ? $obj->list_count : 30;
            $args->page_count = $obj->page_count ? $obj->page_count : 10;

            $output = executeQuery('statistics.getListPages', $args);

            // 결과가 없거나 오류 발생시 그냥 return
            if (!$output->toBool() || !count($output->data)) return new Object();

            return $output;
        }

        function getChartXmlParam($name, $value, $color = 'CCCCCC') {
            return sprintf("<set value='%d' name='%s' color='%s' />", $value, $name, $color);
        }

        function getChartXmlPie($title, $param, $radius = 100) {
            return sprintf(
                "<graph caption='%s' bgColor='FFFFFF' decimalPrecision='0' baseFontSize='11' ".
                "showPercentageValues='0' showPercentageInLabel='1' showNames='1' numberSuffix='' ".
                "showValues='1' formatNumber='1' formatNumberScale='0' thousandSeparator=',' decimalSeparator='.' pieRadius='%d'>".
                "%s</graph>", $title, $radius, $param
            );
        }

    }
?>
